<?php

namespace Drupal\rift\Plugin\Field\FieldFormatter;

use Drupal\Component\Plugin\PluginManagerInterface;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\Plugin\Field\FieldFormatter\EntityReferenceEntityFormatter;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\media\MediaInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;


abstract class RiftPictureFormatterBase extends EntityReferenceEntityFormatter {

  /**
   * Constructs an RiftPictureFormatterBase instance.
   *
   * @param string $plugin_id
   *   The plugin_id for the formatter.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field_definition
   *   The definition of the field to which the formatter is associated.
   * @param array $settings
   *   The formatter settings.
   * @param string $label
   *   The formatter label display setting.
   * @param string $view_mode
   *   The view mode.
   * @param array $third_party_settings
   *   Any third party settings.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $logger_factory
   *   The logger factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityDisplayRepositoryInterface $entity_display_repository
   *   The entity display repository.
   * @param \Drupal\Component\Plugin\PluginManagerInterface $rift_picture_view_modes_manager
   *   The picture view modes manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    $label,
    $view_mode,
    array $third_party_settings,
    LoggerChannelFactoryInterface $logger_factory,
    EntityTypeManagerInterface $entity_type_manager,
    EntityDisplayRepositoryInterface $entity_display_repository,
    protected PluginManagerInterface $riftPictureViewModesManager,
    protected ConfigFactoryInterface $configFactory,
  ) {
    parent::__construct(
      $plugin_id,
      $plugin_definition,
      $field_definition,
      $settings,
      $label,
      $view_mode,
      $third_party_settings,
      $logger_factory,
      $entity_type_manager,
      $entity_display_repository
    );
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['label'],
      $configuration['view_mode'],
      $configuration['third_party_settings'],
      $container->get('logger.factory'),
      $container->get('entity_type.manager'),
      $container->get('entity_display.repository'),
      $container->get('plugin.manager.rift_picture_view_modes'),
      $container->get('config.factory')
    );
  }

  /**
   * Get list of view modes.
   *
   * @return array
   *   The picture view modes as select list options.
   */
  protected function getAvailableViewModes(): array {
    $definitions = $this->riftPictureViewModesManager->getDefinitions();
    $options = [
      '' => $this->t('-Select-'),
    ];
    if (!empty($definitions)) {
      foreach ($definitions as $id => $definition) {
        $options[$id] = $definition['label'] ?? $id;
      }
    }
    return $options;
  }


  /**
   * Builds a rift_picture renderable array for the given entity.
   *
   * @param \Drupal\media\MediaInterface $entity
   *   The entity to be rendered.
   * @param array $config
   *   The configuration array to be applied for rendering.
   * @param string $view_mode
   *   The view mode in which the entity should be rendered.
   *
   * @return array
   *   A renderable array structure suitable for display.
   */
  protected function buildRenderable(MediaInterface $entity, string $view_mode): array {
    $definitions = $this->riftPictureViewModesManager->getDefinitions();
    $config = $definitions[$view_mode] ?? [];
    $config['third_party_settings'] = $this->getThirdPartySettings();
    return [
      '#type' => 'inline_template',
      '#template' => '{{ media|rift_picture(config) }}',
      '#context' => [
        'media' => $entity,
        'config' => $config,
      ],
      '#cache' => [
        'tags' => Cache::mergeTags(
          $entity->getCacheTags(),
          $this->configFactory->get('rift.settings')->getCacheTags()
        ),
        'contexts' => $entity->getCacheContexts(),
        'max-age' => $entity->getCacheMaxAge(),
        'keys' => [
          'entity_view',
          $entity->getEntityTypeId(),
          $entity->id(),
          $view_mode,
        ],
      ],
    ];
  }

}
