<?php

namespace Drupal\rift\Plugin\RiftBuilder;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\media\MediaInterface;
use Drupal\rift\Attribute\RiftBuilder;
use Drupal\rift\DTO\PictureConfig;
use Drupal\rift\DTO\SourceTransformConfig;
use Drupal\rift\Html\ImgElement;
use Drupal\rift\Html\PictureElement;
use Drupal\rift\Html\ElementBase;
use Drupal\rift\Html\SourceElement;
use Drupal\rift\Html\SizesItem;
use Drupal\rift\Html\SrcSetItem;
use Drupal\rift\RiftMediaSourceInterface;
use Drupal\rift\RiftPicture;
use Drupal\rift\RiftBuilderInterface;
use Drupal\rift\RiftSettings;
use Drupal\rift\RiftSourceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Default picture element builder.
 *
 * This is the default implementation that contains the original logic
 * from the RiftPicture service.
 */
#[RiftBuilder(
  id: 'default',
  label: new TranslatableMarkup('Default RIFT Builder'),
  description: new TranslatableMarkup('Default picture element builder that generates standard HTML <picture> element.'),
  priority: 0
)]
class DefaultBuilder implements RiftBuilderInterface, ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static();
  }

  /**
   * {@inheritdoc}
   */
  public function isApplicable(PictureConfig $pictureConfig, ?MediaInterface $media = NULL): bool {
    // Default builder is always applicable as a fallback.
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function build(PictureConfig $pictureConfig, RiftMediaSourceInterface $riftMediaSource, RiftSourceInterface $riftSource, ?MediaInterface $media = NULL): array {
    $picture_element = $this->generatePictureElement($pictureConfig, $riftMediaSource, $riftSource, $media);
    return $picture_element->render();
  }

  /**
   * {@inheritdoc}
   */
  public function getPriority(): int {
    return 0;
  }

  /**
   * Builds picture element DTO object from sources and media.
   *
   * @param \Drupal\rift\DTO\PictureConfig $pictureConfig
   *   The picture configuration object.
   *  @param \Drupal\rift\RiftMediaSourceInterface $riftMediaSource
   *   The Media Source Plugin.
   *  @param \Drupal\rift\RiftSourceInterface $riftSource
   *   The <source> tag generator plugin.
   * @param \Drupal\media\MediaInterface|null $media
   *   The media entity.
   * @param
   *
   * @return \Drupal\rift\Html\ElementBase
   *   The built responsive picture element as render array.
   */
  protected function generatePictureElement(PictureConfig $pictureConfig, RiftMediaSourceInterface $riftMediaSource, RiftSourceInterface $riftSource, ?MediaInterface $media = NULL): ElementBase {
    $picture_element = $this->createPictureElement();
    if (empty($media)) {
      return $picture_element;
    }

    // Check if media has valid image data
    if (!$riftMediaSource->validateMedia($media)) {
      return $picture_element;
    }
    $sources = $this->generatePictureSources($pictureConfig, $media, $riftMediaSource, $riftSource);

    $source_elements = [];

    // Build source elements from sources array.
    foreach ($sources as $source) {
      $image_source = $riftMediaSource->getImageData($media, $source['source_transform_config']);
      // Only add source if it has a valid src
      if (!empty($image_source->getSrc())) {
        $source_elements[] = $this->generateSourceTagFromStyleArray($source, $image_source->getSrc(), $riftMediaSource, $riftSource);
      }
    }
    $picture_element->setSource($source_elements);

    // Generate the <img> tag inside <picture>.
    $image_source = $riftMediaSource->getImageData($media);

    // Only proceed if we have a valid image source
    if (empty($image_source->getSrc())) {
      return $picture_element;
    }

    $dimensions = $this->getImageDimension($image_source, $pictureConfig->fallbackTransform);

    $url = $this->generateImageUrlFromStyles(
      explode('-', $pictureConfig->fallbackTransform),
      $image_source->getSrc(),
      $riftSource
    );

    $img_element = $this->createImgElement();
    $img_element
      ->setSrc($url)
      ->setAlt($image_source->getAlt())
      ->setTitle($image_source->getTitle());

    // Only set height and width if they are not null
    $height = $dimensions['height'] ?: $image_source->getHeight();
    if ($height !== null) {
      $img_element->setHeight($height);
    }

    $width = $dimensions['width'] ?: $image_source->getWidth();
    if ($width !== null) {
      $img_element->setWidth($width);
    }

    $picture_element->setImg($img_element);
    $picture_element->setAttribute($pictureConfig->attribute);

    return $picture_element;
  }

  /**
   * Generate picture sources from a given configuration and media entity.
   *
   * @param \Drupal\rift\PictureConfig $config
   *   Picture configuration.
   * @param \Drupal\media\MediaInterface|null $media
   *   The media entity.
   *
   * @return array
   *   Render array for responsive picture.
   */
  protected function generatePictureSources(PictureConfig $config, MediaInterface $media, RiftMediaSourceInterface $riftMediaSource, RiftSourceInterface $riftSource): array {
    $picture_sources_data = [];
    $image_transforms = [];
    $ndx = 0;
    foreach ($config->sizes as $size) {
      // When aspect ratio is empty for a given size, use "nop".
      $aspect_ratio_style = $config->aspectRatios[$ndx] ?? RiftSettings::NOP;
      [
        $screen,
        $view_width,
        $width,
      ] = $this->extractWidthFromScreenAndViewWidth($size, $config->screens);
      $transform = new SourceTransformConfig(
        size: $size,
        screen: $screen,
        screenWidth: $config->screens[$screen]->width,
        viewWidth:  $view_width,
        width: intval($width),
        mediaQuery: $config->screens[$screen]->mediaQuery,
        effectiveMediaQuery: !isset($config->sizes[$ndx + 1]) ? FALSE : $config->screens[$screen]->mediaQuery,
        aspectRatioStyle:  $aspect_ratio_style,
        transformStyle:  !empty($config->transforms[$ndx]) ? $config->transforms[$ndx] : FALSE,
      );
      $ndx++;
      $image_transforms[] = $transform;
    }

    foreach ($config->formats as $format) {
      foreach ($image_transforms as $transform) {
        $width = $transform->width;
        $srcsets = [];
        foreach ($config->multipliers as $multiplier) {
          $multiplierTransformConfig = clone $transform;
          $quality = $config->quality[$multiplier];
          $m = floatval(trim($multiplier, 'x'));
          $multiplierTransformConfig->width = intval($width * $m);
          $srcset = [
            'styles' => [
              'image_boundaries' => $riftMediaSource->generateImageBoundaryStyle($media, $multiplierTransformConfig),
              'main' => $multiplierTransformConfig->transformStyle,
              'format' => $format == 'original' ? FALSE : $format,
              'quality' => $quality ?: FALSE,
            ],
            'multiplier' => $multiplier,
            'type' => $format == 'original' ? FALSE : 'image/' . $format,
            'source_transform_config' => $multiplierTransformConfig,
          ];
          $srcsets[] = $srcset;
        }
        $picture_sources_data[] = [
          'media' => $transform->effectiveMediaQuery,
          'sizes' => $transform->viewWidth,
          'view_width' => $transform->viewWidth,
          'srcset' => $srcsets,
          'width' => $transform->width,
          'source_transform_config' => $transform,
        ];
      }
    }
    return $picture_sources_data;
  }

  /**
   * Generate <source> tag from style definitions.
   *
   * @param mixed $config
   *   Configurations from which a source tag can be generated.
   * @param string|null $uri
   *   The uri for source tag.
   * @param \Drupal\rift\RiftPicture $riftPicture
   *   The RiftPicture service instance.
   *
   * @return \Drupal\rift\Html\SourceElement
   *   Returns source tag render array.
   */
  protected function generateSourceTagFromStyleArray(mixed $config, ?string $uri, RiftMediaSourceInterface $riftMediaSource, RiftSourceInterface $riftSource): SourceElement {
    $srcset_items = [];
    $source_element = $this->createSourceElement();
    foreach ($config['srcset'] as $source) {
      $url = $this->generateImageUrlFromStyles($source['styles'], $uri, $riftSource);
      $srcset_item = $this->createSrcSetItem();
      $srcset_item->setImg($url);
      // Unlike traditional width, the width in srcset item is the multiplier.
      $srcset_item->setWidth($source['multiplier']);
      $srcset_items[] = $srcset_item;
      // We set the type to the last one in the source.
      // Is this good idea?
      if ($source['type']) {
        $source_element->setType($source['type']);
      }
      if (!$source_element->getHeight()) {
        if ($width = $source['source_transform_config']->width) {
          if ($source['source_transform_config']->aspectRatioStyle !== RiftSettings::NOP) {
            [$arw, $arh] = explode('x', $source['source_transform_config']->aspectRatioStyle);
            if ($arh > 0) {
              $aspect_ratio = intval($arw) / intval($arh);
              $height = intval($width / $aspect_ratio);
              if ($height) {
                $source_element->setWidth($width);
                $source_element->setHeight($height);
              }
            }
          }
        }
      }
    }
    $sizes_item = $this->createSizesItem();
    $sizes_item->setMediaQuery($config['media']);
    $sizes_item->setWidth($config['view_width']);
    $source_element
      ->setMedia($config['media'])
      ->setSizes([$sizes_item])
      ->setSrcset($srcset_items);
    return $source_element;
  }

  /**
   * Finds height and width of an image.
   *
   * @param \Drupal\rift\Html\ImgElement $image_source
   *   The image source.
   * @param string|null $combined_image_style
   *   The combined image style.
   *
   * @return array
   *   Dimension of the image.
   */
  protected function getImageDimension(ImgElement $image_source, ?string $combined_image_style): array {
    $url = $image_source->getSrc();
    $dimensions = [
      'width' => $image_source->getWidth(),
      'height' => $image_source->getHeight(),
    ];
    foreach (explode('-', $combined_image_style) as $style) {
      // phpcs:ignore DrupalPractice.Objects.GlobalClass.GlobalClass
      $image_style = \Drupal\image\Entity\ImageStyle::load($style);
      if ($image_style) {
        $image_style->transformDimensions($dimensions, $url);
      }
    }
    return $dimensions;
  }

  /**
   * Generate Image url from image styles definitions.
   */
  public function generateImageUrlFromStyles(array $input_styles, ?string $uri, RiftSourceInterface $riftSource): string {
    return $riftSource
      ->generate($input_styles, $uri);
  }


  /**
   * Helper method to extract width from screen and view width definition.
   *
   * @param string $size
   *   Size definition (E.g. sm:100vw, lg:400px, ...).
   * @param \Drupal\rift\DTO\ScreenConfig[] $screens
   *   Screen configurations.
   *
   * @return array
   *   Returns screen, view_width definition and width.
   *
   * @todo fix return type.
   */
  protected function extractWidthFromScreenAndViewWidth(string $size, array $screens): array {
    [$screen, $view_width] = explode(':', $size);
    $screen_width = $screens[$screen]->width;
    $view_width = trim($view_width, 'px');
    if (str_ends_with($view_width, 'vw')) {
      $percent = floatval(trim($view_width, 'vw'));
      $width = intval($screen_width * $percent / 100);
    }
    else {
      $width = $view_width;
    }
    return [$screen, $view_width, $width];
  }


  /**
   * Creates a new img element DTO object.
   *
   * @return \Drupal\rift\Html\ImgElement
   *   Returns a new img element instance.
   */
  protected function createImgElement(): ImgElement {
    return new ImgElement();
  }

  /**
   * Creates a new picture element DTO object.
   *
   * @return \Drupal\rift\Html\PictureElement
   *   Returns a new picture element instance.
   */
  protected function createPictureElement(): PictureElement {
    return new PictureElement();
  }

  /**
   * Creates a new sizes item DTO object.
   *
   * @return \Drupal\rift\Html\SizesItem
   *   Returns a new sizes item instance.
   */
  protected function createSizesItem(): SizesItem {
    return new SizesItem();
  }

  /**
   * Creates a new source element DTO object.
   *
   * @return \Drupal\rift\Html\SourceElement
   *   Returns a new source element instance.
   */
  protected function createSourceElement(): SourceElement {
    return new SourceElement();
  }

  /**
   * Creates a new srcset item DTO object.
   *
   * @return \Drupal\rift\Html\SrcSetItem
   *   Returns a new srcset item instance.
   */
  protected function createSrcSetItem(): SrcSetItem {
    return new SrcSetItem();
  }

}
