<?php

namespace Drupal\rift;

use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\media\MediaInterface;
use Drupal\rift\DTO\PictureConfig;
use Drupal\rift\Html\ElementBase;
use Drupal\rift\Html\PictureElement;

/**
 * Plugin manager for RiftBuilder plugins.
 */
class RiftBuilderManager extends DefaultPluginManager {

  /**
   * Constructs a RiftBuilderManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler to invoke the alter hook with.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct(
      'Plugin/RiftBuilder',
      $namespaces,
      $module_handler,
      RiftBuilderInterface::class,
      'Drupal\rift\Attribute\RiftBuilder'
    );

    $this->alterInfo('rift_picture_element_builder_info');
    $this->setCacheBackend($cache_backend, 'rift_picture_element_builder_plugins');
  }

  /**
   * Builds picture element markup using the appropriate builder.
   *
   * @param \Drupal\rift\DTO\PictureConfig $pictureConfig
   *   The picture configuration object.
   *  @param \Drupal\rift\RiftMediaSourceInterface $riftMediaSource
   *   The Media Source Plugin.
   *  @param \Drupal\rift\RiftSourceInterface $riftSource
   *   The <source> tag generator plugin.
   * @param \Drupal\media\MediaInterface|null $media
   *   The media entity.
   * @param
   *
   * @return \Drupal\rift\Html\ElementBase
   *   The built responsive picture element as render array.
   */
  public function buildMarkup(PictureConfig $pictureConfig, RiftMediaSourceInterface $riftMediaSource, RiftSourceInterface $riftSource, ?MediaInterface $media): array {

    $builders = $this->getDefinitions();

    // Sort builders by priority (highest first).
    uasort($builders, function ($a, $b) {
      $priority_a = $a['priority'] ?? 0;
      $priority_b = $b['priority'] ?? 0;
      return $priority_b <=> $priority_a;
    });

    // Find the first applicable builder.
    foreach ($builders as $plugin_id => $definition) {
      $builder = $this->createInstance($plugin_id);
      if ($builder->isApplicable($pictureConfig, $media)) {
        return $builder->build($pictureConfig, $riftMediaSource, $riftSource, $media);
      }
    }

    // Fallback to default builder if no applicable builder found.
    return $this->createInstance('default')->build($pictureConfig, $riftMediaSource, $riftSource, $media);
  }

}
