<?php

namespace Drupal\rift;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Render\Element;
use Drupal\Core\Render\RendererInterface;
use Drupal\media\MediaInterface;
use Drupal\rift\DTO\PictureConfig;
use Drupal\rift\DTO\SourceTransformConfig;
use Drupal\rift\Html\ImgElement;
use Drupal\rift\Html\PictureElement;
use Drupal\rift\Html\SizesItem;
use Drupal\rift\Html\SourceElement;
use Drupal\rift\Html\SrcSetItem;
use Drupal\image\Entity\ImageStyle;
use Drupal\search_api\Plugin\search_api\processor\Resources\Pi;

/**
 * Service to generate responsive picture markup from a media entity.
 */
class RiftPicture {

  /**
   * Constructs the TwigTweakExtension object.
   */
  public function __construct(
    private readonly RendererInterface $renderer,
    private readonly ConfigFactoryInterface $configFactory,
    private readonly RiftSourceManager $riftSource,
    private readonly RiftMediaSourceManager $riftMediaSource,
    private readonly RiftBuilderManager $riftBuilder,
  ) {}

  /**
   * Get the default "media source" plugin ID.
   *
   * @return string
   *   The media source plugin id.
   */
  protected function getDefaultMediaSourcePluginId() : string {
    return $this->configFactory->get('rift.settings')->get('media_source') ?: 'image';
  }

  /**
   * Get the default "source" plugin ID.
   *
   * @return string
   *   The source plugin id.
   */
  protected function getDefaultSourcePluginId() : string {
    return $this->configFactory->get('rift.settings')->get('source') ?: 'combined_image_style';
  }


  protected function loadRiftMediaSourceProcessor(string $plugin_id = NULL): RiftMediaSourceInterface {
    if (empty($plugin_id)) {
      $plugin_id = $this->getDefaultMediaSourcePluginId();
    }
    /** @var \Drupal\rift\RiftMediaSourceInterface $plugin */
    $plugin = $this->riftMediaSource->createInstance($plugin_id);
    return $plugin;
  }

  protected function loadRiftSourceProcessor(string $plugin_id = NULL): RiftSourceInterface {
    if (empty($plugin_id)) {
      $plugin_id = $this->getDefaultSourcePluginId();
    }
    /** @var \Drupal\rift\RiftSourceInterface $plugin */
    $plugin = $this->riftSource->createInstance($plugin_id);
    return $plugin;
  }

  /**
   * Get sitewide default configurations.
   *
   * @return array
   *   The default configurations.
   */
  protected function getDefaultConfig(): array {
    $config = $this->configFactory->get('rift.settings')->get('config');
    $config['url_generation_strategy'] = $this->getDefaultSourcePluginId();
    $config['third_party_settings'] = [];
    return $config;
  }

  /**
   * Responsive Picture twig filter.
   *
   * @param \Drupal\media\MediaInterface $media
   *   Media entity.
   * @param array $config
   *   Configuration for generating srcset.
   *   <code>
   *    node.field_common_image.0.entity|rift_picture({
   *      sizes: 'xs:100vw sm:100vw md:50vw lg:50vw xl:50vw',
   *      aspect_ratios: "1x1 16x9 1x1 1x1 1x1",
   *      quality: {
   *        "1x": "80",
   *        "2x": "40",
   *      },
   *      url_generation_strategy: "dummyimage",
   *      formats: ['webp', 'avif', 'jpeg'],
   *    })
   *   </code>.
   *
   * @return string[]
   *   Render array for responsive picture tag.
   *
   * @throws \Drupal\Core\TypedData\Exception\MissingDataException
   */
  public function responsivePicture(?MediaInterface $media = NULL, array $config = []): array {
    $config = array_merge($this->getDefaultConfig(), $config);
    if (empty($config['id'])) {
      $config['id'] = '#' . md5(serialize($config));
    }
    $picture_config = new PictureConfig();
    $picture_config
      ->processScreens($config['screens'])
      ->processSizes($config['sizes'])
      ->processTransforms($config['transforms'])
      ->processMultipliers($config['multipliers'])
      ->processQuality($config['quality'])
      ->processAspectRatios($config['aspect_ratios'])
      ->processFormats($config['formats'])
      ->processAttribute($config['attributes'])
      ->processFallbackTransforms($config['fallback_transform'])
      ->processUrlGenerationStrategy($config['url_generation_strategy'])
      ->processThirdPartySettings($config['third_party_settings']);

    // Use the plugin system to build the picture element.
    $picture_element = $this->riftBuilder->buildMarkup(
      $picture_config,
      $this->loadRiftMediaSourceProcessor(),
      $this->loadRiftSourceProcessor($picture_config->urlGenerationStrategy),
      $media
    );

    // We avoid using dependency injection to avoid circular dependency.
    // phpcs:ignore DrupalPractice.Objects.GlobalDrupal.GlobalDrupal
    if (\Drupal::service('twig')->isDebug()) {
      $output = [
        [
          '#type' => 'inline_template',
          '#template' => '<!-- RIFT View Mode : {{ view_mode }} -->',
          '#context' => [
            'view_mode' => $config['id'],
          ],
        ],
        $picture_element,
      ];
    }
    else {
      $output = $picture_element;
    }
    return $output;
  }

}
