<?php

namespace Drupal\rift\ExistingSite;

use Drupal\Core\Template\Attribute;
use Drupal\file\Entity\File;
use Drupal\Tests\TestFileCreationTrait;
use weitzman\DrupalTestTraits\Entity\MediaCreationTrait;
use weitzman\DrupalTestTraits\ExistingSiteBase;

/**
 * Basic tests for Rift Picture functionality.
 */
class RiftPictureBasicTest extends ExistingSiteBase {
  use MediaCreationTrait;
  use TestFileCreationTrait;

  /**
   * The sample image File entity to embed.
   *
   * @var \Drupal\media\MediaInterface
   */
  protected $media;

  /**
   * The sample image File entity to embed.
   *
   * @var \Drupal\file\FileInterface
   */
  protected $file;

  /**
   * URI of the image file being used for testing.
   *
   * @var string
   */
  protected $imageFileUri;

  /**
   * {@inheritdoc}
   */
  protected function setUp() : void {
    parent::setUp();
    $this->imageFileUri = $this->getTestFiles('image')[0]->uri;
    // Create a sample host entity to embed images in.
    $this->file = File::create([
      'uri' => $this->imageFileUri,
    ]);
    $this->file
      ->save();
    $this->markEntityForCleanup($this->file);
    $this->media = $this->createMedia(
      [
        'bundle' => 'image',
        'name' => 'test_name_test_name_test_name',
        'field_media_image' => [
          'target_id' => $this->file->id(),
          'alt' => 'test_alt_test_alt_test_alt',
          'title' => 'test_title_test_title_test_title',
        ],
      ]
    );
    $this->markEntityForCleanup($this->media);
  }

  /**
   * Test case for media with empty bundle.
   */
  public function testEmptyImageInMedia() {
    $media = $this->createMedia(['bundle' => 'image']);
    $this->markEntityForCleanup($media);
    $service = \Drupal::service('rift.picture');
    $config = $this->getBaseConfig();
    $elements = $service->responsivePicture($media, $config);
    /** @var \Drupal\Core\Render\Renderer $renderer */
    $renderer = \Drupal::service('renderer');
    $string = $renderer->renderInIsolation($elements);
    // Should return an empty picture element when no image data is available
    $this->assertEquals('<picture ></picture>', $string);
  }

  /**
   * Test responsive picture basic functionality.
   */
  public function testResponsivePicture() {
    $media = $this->media;
    $service = \Drupal::service('rift.picture');
    $config = $this->getBaseConfig();
    $elements = $service->responsivePicture($media, $config);
    /** @var \Drupal\Core\Render\Renderer $renderer */
    $renderer = \Drupal::service('renderer');
    $string = $renderer->renderInIsolation($elements);
    $this->assertNotEmpty($string, 'Expected image markup to be present, but empty markup returned.');

    $picture_attribute = $elements['#context']['attributes'];
    $this->assertInstanceOf(Attribute::class, $picture_attribute);
    $this->assertEquals('lazy', $picture_attribute->offsetGet('loading'));
    $this->assertEquals('', $picture_attribute->offsetGet('preload'));

    $this->assertEquals(10, count($elements['#context']['content']));
    $this->assertEquals('picture', $elements['#context']['tag_name']);
    $this->assertEquals('source', $elements['#context']['content'][0]['#context']['tag_name']);
  }

  /**
   * Test "rift_picture" Twig Filter.
   */
  public function testResponsivePictureTwigFilter() {
    $file = File::create([
      'uri' => $this->getTestFiles('image')[0]->uri,
    ]);
    $file->save();
    $media = $this->createMedia([
      'bundle' => 'image',
      'field_media_image' => [
        'target_id' => $file->id(),
        'title' => 'dummy title',
        'alt' => 'dummy alt',
      ],
    ]);
    $this->markEntityForCleanup($file);
    $this->markEntityForCleanup($media);
    $elements = [
      '#type' => 'inline_template',
      '#template' => '{{ media|rift_picture({
          sizes: "sm:100w",
          aspect_ratios: "1x2",
          url_generation_strategy: "placeholdit",
        })
      }}',
      '#context' => [
        'media' => $media,
      ],
    ];
    /** @var \Drupal\Core\Render\Renderer $renderer */
    $renderer = \Drupal::service('renderer');
    $output = $renderer->renderInIsolation($elements);
    $html = '<picture  loading="lazy" preload=""><source  width="100" height="200" type="image/webp" sizes="100w" srcset="https://place-hold.it/100x200?text=100w200h-webp-80 1x,https://place-hold.it/200x400?text=200w400h-webp-40 2x" /><source  width="100" height="200" type="image/jpeg" sizes="100w" srcset="https://place-hold.it/100x200?text=100w200h-jpeg-80 1x,https://place-hold.it/200x400?text=200w400h-jpeg-40 2x" /><img  src="https://place-hold.it/x?text=medium-80" alt="dummy alt" title="dummy title" width="40" height="20" /></picture>';
    $this->assertEquals($html, $output);
  }

  /**
   * Get default configuration to be used for testing.
   */
  protected function getBaseConfig() {
    return [
      'sizes' => 'md:100 lg:300 xl:500',
      'aspect_ratios' => '1x1 1x1 1x1',
      'screens' => [
        'xs' => [
          'width' => 640,
          'media' => '(max-width: 640px)',
        ],
        'sm' => [
          'width' => 768,
          'media' => '(max-width: 768px)',
        ],
        'md' => [
          'width' => 1024,
          'media' => '(max-width: 1024px)',
        ],
        'lg' => [
          'width' => 1280,
          'media' => '(max-width: 1280px)',
        ],
        'xl' => [
          'width' => 1536,
          'media' => '',
        ],
      ],
      'transforms' => '',
      'quality' => [
        '1x' => 80,
        '2x' => 40,
      ],
      'formats' => [
        'avif',
        'webp',
        'jpeg',
      ],
      'attributes' => [
        'loading' => 'lazy',
        'preload' => '',
      ],
      'fallback_transform' => 'medium-80',
      'url_generation_strategy' => 'combined_image_style',
      'multipliers' => [
        '1x',
        '2x',
      ],
    ];
  }

} 