<?php

namespace Drupal\rift\ExistingSite;

use Drupal\file\Entity\File;
use Drupal\Tests\TestFileCreationTrait;
use weitzman\DrupalTestTraits\Entity\MediaCreationTrait;
use weitzman\DrupalTestTraits\ExistingSiteBase;

/**
 * Tests for Rift Picture configuration options.
 */
class RiftPictureConfigurationTest extends ExistingSiteBase {
  use MediaCreationTrait;
  use TestFileCreationTrait;

  /**
   * The sample image File entity to embed.
   *
   * @var \Drupal\media\MediaInterface
   */
  protected $media;

  /**
   * The sample image File entity to embed.
   *
   * @var \Drupal\file\FileInterface
   */
  protected $file;

  /**
   * URI of the image file being used for testing.
   *
   * @var string
   */
  protected $imageFileUri;

  /**
   * {@inheritdoc}
   */
  protected function setUp() : void {
    parent::setUp();
    $this->imageFileUri = $this->getTestFiles('image')[0]->uri;
    // Create a sample host entity to embed images in.
    $this->file = File::create([
      'uri' => $this->imageFileUri,
    ]);
    $this->file
      ->save();
    $this->markEntityForCleanup($this->file);
    $this->media = $this->createMedia(
      [
        'bundle' => 'image',
        'name' => 'test_name_test_name_test_name',
        'field_media_image' => [
          'target_id' => $this->file->id(),
          'alt' => 'test_alt_test_alt_test_alt',
          'title' => 'test_title_test_title_test_title',
        ],
      ]
    );
    $this->markEntityForCleanup($this->media);
  }

  /**
   * Test Aspect Ratio configuration for "nop".
   */
  public function testAspectRatioConfigWithNop() {
    $media = $this->media;
    $service = \Drupal::service('rift.picture');
    $config = $this->getBaseConfig();
    $config['aspect_ratios'] = '2x3 24x10 nop';
    $config['url_generation_strategy'] = 'dummyimage';
    $elements = $service->responsivePicture($media, $config);
    $this->assertEquals(10, count($elements['#context']['content']));
    $map = [
      0 => [
        'https://dummyimage.com/100x150/eee/222?text=100w150h-avif-80',
        'https://dummyimage.com/200x300/eee/222?text=200w300h-avif-40',
      ],
      1 => [
        'https://dummyimage.com/300x125/eee/222?text=300w125h-avif-80',
        'https://dummyimage.com/600x250/eee/222?text=600w250h-avif-40',
      ],
      // For aspect ratio "nop" we would only scale width.
      2 => [
        'https://dummyimage.com/500x/eee/222?text=500w-avif-80',
        'https://dummyimage.com/1000x/eee/222?text=1000w-avif-40',
      ],
      3 => [
        'https://dummyimage.com/100x150/eee/222?text=100w150h-webp-80',
        'https://dummyimage.com/200x300/eee/222?text=200w300h-webp-40',
      ],
      4 => [
        'https://dummyimage.com/300x125/eee/222?text=300w125h-webp-80',
        'https://dummyimage.com/600x250/eee/222?text=600w250h-webp-40',
      ],
      // For aspect ratio "nop" we would only scale width.
      5 => [
        'https://dummyimage.com/500x/eee/222?text=500w-webp-80',
        'https://dummyimage.com/1000x/eee/222?text=1000w-webp-40',
      ],
      6 => [
        'https://dummyimage.com/100x150/eee/222?text=100w150h-jpeg-80',
        'https://dummyimage.com/200x300/eee/222?text=200w300h-jpeg-40',
      ],
      7 => [
        'https://dummyimage.com/300x125/eee/222?text=300w125h-jpeg-80',
        'https://dummyimage.com/600x250/eee/222?text=600w250h-jpeg-40',
      ],
      // For aspect ratio "nop" we would only scale width.
      8 => [
        'https://dummyimage.com/500x/eee/222?text=500w-jpeg-80',
        'https://dummyimage.com/1000x/eee/222?text=1000w-jpeg-40',
      ],
    ];
    foreach ($elements["#context"]["content"] as $ndx => $content) {
      /** @var \Drupal\Core\Template\Attribute $attributes */
      $attribute = $content['#context']['attributes'];
      if ($content['#context']['tag_name'] == "source") {
        $srcset_string = (string) $attribute->offsetGet('srcset');
        $srcset_items = explode(',', $srcset_string);
        $this->assertEquals(2, count($srcset_items));
        [$first_url] = explode(' ', $srcset_items[0]);
        [$second_url] = explode(' ', $srcset_items[1]);
        $this->assertEquals($map[$ndx][0], $first_url);
        $this->assertEquals($map[$ndx][1], $second_url);
      }
    }
  }


  /**
   * Test Aspect Ratio configuration when aspect ratio is missing.
   */
  public function testMissingAspectRatioConfig() {
    $media = $this->media;
    $service = \Drupal::service('rift.picture');
    $config = $this->getBaseConfig();
    $config['aspect_ratios'] = '2x3 24x10';
    $config['url_generation_strategy'] = 'dummyimage';
    $elements = $service->responsivePicture($media, $config);
    $this->assertEquals(10, count($elements['#context']['content']));
    $map = [
      0 => [
        'https://dummyimage.com/100x150/eee/222?text=100w150h-avif-80',
        'https://dummyimage.com/200x300/eee/222?text=200w300h-avif-40',
      ],
      1 => [
        'https://dummyimage.com/300x125/eee/222?text=300w125h-avif-80',
        'https://dummyimage.com/600x250/eee/222?text=600w250h-avif-40',
      ],
      // For aspect ratio "nop" we would only scale width.
      2 => [
        'https://dummyimage.com/500x/eee/222?text=500w-avif-80',
        'https://dummyimage.com/1000x/eee/222?text=1000w-avif-40',
      ],
      3 => [
        'https://dummyimage.com/100x150/eee/222?text=100w150h-webp-80',
        'https://dummyimage.com/200x300/eee/222?text=200w300h-webp-40',
      ],
      4 => [
        'https://dummyimage.com/300x125/eee/222?text=300w125h-webp-80',
        'https://dummyimage.com/600x250/eee/222?text=600w250h-webp-40',
      ],
      // For aspect ratio "nop" we would only scale width.
      5 => [
        'https://dummyimage.com/500x/eee/222?text=500w-webp-80',
        'https://dummyimage.com/1000x/eee/222?text=1000w-webp-40',
      ],
      6 => [
        'https://dummyimage.com/100x150/eee/222?text=100w150h-jpeg-80',
        'https://dummyimage.com/200x300/eee/222?text=200w300h-jpeg-40',
      ],
      7 => [
        'https://dummyimage.com/300x125/eee/222?text=300w125h-jpeg-80',
        'https://dummyimage.com/600x250/eee/222?text=600w250h-jpeg-40',
      ],
      // For aspect ratio "nop" we would only scale width.
      8 => [
        'https://dummyimage.com/500x/eee/222?text=500w-jpeg-80',
        'https://dummyimage.com/1000x/eee/222?text=1000w-jpeg-40',
      ],
    ];
    foreach ($elements["#context"]["content"] as $ndx => $content) {
      /** @var \Drupal\Core\Template\Attribute $attributes */
      $attribute = $content['#context']['attributes'];
      if ($content['#context']['tag_name'] == "source") {
        $srcset_string = (string) $attribute->offsetGet('srcset');
        $srcset_items = explode(',', $srcset_string);
        $this->assertEquals(2, count($srcset_items));
        [$first_url] = explode(' ', $srcset_items[0]);
        [$second_url] = explode(' ', $srcset_items[1]);
        $this->assertEquals($map[$ndx][0], $first_url);
        $this->assertEquals($map[$ndx][1], $second_url);
      }
    }
  }


  /**
   * Test Aspect Ratio configuration.
   */
  public function testAspectRatioConfig() {
    $media = $this->media;
    $service = \Drupal::service('rift.picture');
    $config = $this->getBaseConfig();
    $config['aspect_ratios'] = '2x3 24x10 100x185';
    $config['url_generation_strategy'] = 'dummyimage';
    $elements = $service->responsivePicture($media, $config);
    $this->assertEquals(10, count($elements['#context']['content']));
    $map = [
      0 => [
        'https://dummyimage.com/100x150/eee/222?text=100w150h-avif-80',
        'https://dummyimage.com/200x300/eee/222?text=200w300h-avif-40',
      ],
      1 => [
        'https://dummyimage.com/300x125/eee/222?text=300w125h-avif-80',
        'https://dummyimage.com/600x250/eee/222?text=600w250h-avif-40',
      ],
      2 => [
        'https://dummyimage.com/500x925/eee/222?text=500w925h-avif-80',
        'https://dummyimage.com/1000x1850/eee/222?text=1000w1850h-avif-40',
      ],
      3 => [
        'https://dummyimage.com/100x150/eee/222?text=100w150h-webp-80',
        'https://dummyimage.com/200x300/eee/222?text=200w300h-webp-40',
      ],
      4 => [
        'https://dummyimage.com/300x125/eee/222?text=300w125h-webp-80',
        'https://dummyimage.com/600x250/eee/222?text=600w250h-webp-40',
      ],
      5 => [
        'https://dummyimage.com/500x925/eee/222?text=500w925h-webp-80',
        'https://dummyimage.com/1000x1850/eee/222?text=1000w1850h-webp-40',
      ],
      6 => [
        'https://dummyimage.com/100x150/eee/222?text=100w150h-jpeg-80',
        'https://dummyimage.com/200x300/eee/222?text=200w300h-jpeg-40',
      ],
      7 => [
        'https://dummyimage.com/300x125/eee/222?text=300w125h-jpeg-80',
        'https://dummyimage.com/600x250/eee/222?text=600w250h-jpeg-40',
      ],
      8 => [
        'https://dummyimage.com/500x925/eee/222?text=500w925h-jpeg-80',
        'https://dummyimage.com/1000x1850/eee/222?text=1000w1850h-jpeg-40',
      ],
    ];
    foreach ($elements["#context"]["content"] as $ndx => $content) {
      /** @var \Drupal\Core\Template\Attribute $attributes */
      $attribute = $content['#context']['attributes'];
      if ($content['#context']['tag_name'] == "source") {
        $srcset_string = (string) $attribute->offsetGet('srcset');
        $srcset_items = explode(',', $srcset_string);
        $this->assertEquals(2, count($srcset_items));
        [$first_url] = explode(' ', $srcset_items[0]);
        [$second_url] = explode(' ', $srcset_items[1]);
        $this->assertEquals($map[$ndx][0], $first_url);
        $this->assertEquals($map[$ndx][1], $second_url);
      }
    }
  }

  /**
   * Test responsive picture format support.
   */
  public function testResponsivePictureFormats() {
    $media = $this->media;
    $service = \Drupal::service('rift.picture');
    $config = $this->getBaseConfig();
    $config['formats'] = ['jpeg'];
    $elements = $service->responsivePicture($media, $config);
    $this->assertEquals(4, count($elements['#context']['content']));
    $config['formats'] = ['avif', 'jpeg'];
    $elements = $service->responsivePicture($media, $config);
    $this->assertEquals(7, count($elements['#context']['content']));
    $config['formats'] = ['avif', 'webp', 'jpeg'];
    $elements = $service->responsivePicture($media, $config);
    $this->assertEquals(10, count($elements['#context']['content']));
  }

  /**
   * Test responsive picture 'multipliers' support.
   */
  public function testResponsivePictureMultipliers() {
    $media = $this->media;
    $service = \Drupal::service('rift.picture');
    $config = $this->getBaseConfig();
    $config['multipliers'] = ['1x'];
    $elements = $service->responsivePicture($media, $config);
    $this->assertEquals(10, count($elements['#context']['content']));
    foreach ($elements["#context"]["content"] as $content) {
      /** @var \Drupal\Core\Template\Attribute $attributes */
      $attribute = $content['#context']['attributes'];
      if ($content['#context']['tag_name'] == "source") {
        $srcset_string = (string) $attribute->offsetGet('srcset');
        $srcset_items = explode(',', $srcset_string);
        $this->assertEquals(1, count($srcset_items));
        [, $first_multiplier] = explode(' ', $srcset_items[0]);
        $this->assertEquals('1x', $first_multiplier);
      }
    }

    $config['multipliers'] = ['1x', '2x'];
    $elements = $service->responsivePicture($media, $config);
    $this->assertEquals(10, count($elements['#context']['content']));
    foreach ($elements["#context"]["content"] as $content) {
      /** @var \Drupal\Core\Template\Attribute $attributes */
      $attribute = $content['#context']['attributes'];
      if ($content['#context']['tag_name'] == "source") {
        $srcset_string = (string) $attribute->offsetGet('srcset');
        $srcset_items = explode(',', $srcset_string);
        $this->assertEquals(2, count($srcset_items));
        [, $first_multiplier] = explode(' ', $srcset_items[0]);
        [, $second_multiplier] = explode(' ', $srcset_items[1]);
        $this->assertEquals('1x', $first_multiplier);
        $this->assertEquals('2x', $second_multiplier);
      }
    }
  }

  /**
   * Get default configuration to be used for testing.
   */
  protected function getBaseConfig() {
    return [
      'sizes' => 'md:100 lg:300 xl:500',
      'aspect_ratios' => '1x1 1x1 1x1',
      'screens' => [
        'xs' => [
          'width' => 640,
          'media' => '(max-width: 640px)',
        ],
        'sm' => [
          'width' => 768,
          'media' => '(max-width: 768px)',
        ],
        'md' => [
          'width' => 1024,
          'media' => '(max-width: 1024px)',
        ],
        'lg' => [
          'width' => 1280,
          'media' => '(max-width: 1280px)',
        ],
        'xl' => [
          'width' => 1536,
          'media' => '',
        ],
      ],
      'transforms' => '',
      'quality' => [
        '1x' => 80,
        '2x' => 40,
      ],
      'formats' => [
        'avif',
        'webp',
        'jpeg',
      ],
      'attributes' => [
        'loading' => 'lazy',
        'preload' => '',
      ],
      'fallback_transform' => 'medium-80',
      'url_generation_strategy' => 'combined_image_style',
      'multipliers' => [
        '1x',
        '2x',
      ],
    ];
  }

}
