<?php

namespace Drupal\rift\ExistingSite;

use Drupal\Core\Template\Attribute;
use Drupal\file\Entity\File;
use Drupal\Tests\TestFileCreationTrait;
use weitzman\DrupalTestTraits\Entity\MediaCreationTrait;
use weitzman\DrupalTestTraits\ExistingSiteBase;

/**
 * Detailed tests for Rift Picture element structure and attributes.
 */
class RiftPictureDetailedTest extends ExistingSiteBase {
  use MediaCreationTrait;
  use TestFileCreationTrait;

  /**
   * The sample image File entity to embed.
   *
   * @var \Drupal\media\MediaInterface
   */
  protected $media;

  /**
   * The sample image File entity to embed.
   *
   * @var \Drupal\file\FileInterface
   */
  protected $file;

  /**
   * URI of the image file being used for testing.
   *
   * @var string
   */
  protected $imageFileUri;

  /**
   * {@inheritdoc}
   */
  protected function setUp() : void {
    parent::setUp();
    $this->imageFileUri = $this->getTestFiles('image')[0]->uri;
    // Create a sample host entity to embed images in.
    $this->file = File::create([
      'uri' => $this->imageFileUri,
    ]);
    $this->file
      ->save();
    $this->markEntityForCleanup($this->file);
    $this->media = $this->createMedia(
      [
        'bundle' => 'image',
        'name' => 'test_name_test_name_test_name',
        'field_media_image' => [
          'target_id' => $this->file->id(),
          'alt' => 'test_alt_test_alt_test_alt',
          'title' => 'test_title_test_title_test_title',
        ],
      ]
    );
    $this->markEntityForCleanup($this->media);
  }

  /**
   * Test detailed responsive picture element structure.
   */
  public function testDetailedResponsivePicture() {
    $media = $this->media;
    $service = \Drupal::service('rift.picture');
    $config = $this->getBaseConfig();
    $elements = $service->responsivePicture($media, $config);
    /** @var \Drupal\Core\Render\Renderer $renderer */
    $renderer = \Drupal::service('renderer');
    $string = $renderer->renderInIsolation($elements);
    $this->assertNotEmpty($string, 'Expected image markup to be present, but empty markup returned.');

    $picture_attribute = $elements['#context']['attributes'];
    $this->assertInstanceOf(Attribute::class, $picture_attribute);
    $this->assertEquals('lazy', $picture_attribute->offsetGet('loading'));
    $this->assertEquals('', $picture_attribute->offsetGet('preload'));

    $this->assertEquals(10, count($elements['#context']['content']));
    $this->assertEquals('picture', $elements['#context']['tag_name']);
    $this->assertEquals('source', $elements['#context']['content'][0]['#context']['tag_name']);

    foreach ($elements['#context']['content'] as $position => $content) {
      $attribute = $content['#context']['attributes'];
      $this->assertInstanceOf(Attribute::class, $attribute);
      // Test correct 'tag' is used.
      switch ($position) {
        case 0:
        case 1:
        case 2:
        case 3:
        case 4:
        case 5:
        case 6:
        case 7:
        case 8:
          $this->assertEquals('source', $content['#context']['tag_name']);
          break;

        case 9:
          $this->assertEquals('img', $content['#context']['tag_name'], 'Expected to find an img tag as last fallback item inside picture, but found ' . $content['#context']['tag_name']);
          break;

        default:
          $this->error('Un-expected content detected inside <picture> element.');
          break;
      }
      // Test attribute has correct 'type'.
      switch ($position) {
        case 0:
        case 1:
        case 2:
          $this->assertEquals('image/avif', $attribute->offsetGet('type'));
          break;

        case 3:
        case 4:
        case 5:
          $this->assertEquals('image/webp', $attribute->offsetGet('type'));
          break;

        case 6:
        case 7:
        case 8:
          $this->assertEquals('image/jpeg', $attribute->offsetGet('type'));
          break;

        case 9:
          $this->assertNull($attribute->offsetGet('type'));
          break;

        default:
          $this->error('Un-expected content detected inside <picture> element.');
          break;
      }
      // Test attribute has correct 'media'.
      switch ($position) {
        case 0:
        case 3:
        case 6:
          $this->assertEquals('(max-width: 1024px)', $attribute->offsetGet('media'));
          break;

        case 1:
        case 4:
        case 7:
          $this->assertEquals('(max-width: 1280px)', $attribute->offsetGet('media'));
          break;

        case 2:
        case 5:
        case 8:
          $this->assertNull($attribute->offsetGet('media'), 'Expected no media query to be present on last set of images, but found a media attribute to be present.');
          break;

        case 9:
          $this->assertNull($attribute->offsetGet('media'), 'Expected no media query to be present on fallback img element, but found a media attribute to be present.');
          break;

        default:
          $this->error('Un-expected content detected inside <picture> element.');
          break;
      }

      // Test attribute has correct 'sizes'.
      switch ($position) {
        case 0:
        case 3:
        case 6:
          $this->assertEquals('(max-width: 1024px) 100', $attribute->offsetGet('sizes'));
          break;

        case 1:
        case 4:
        case 7:
          $this->assertEquals('(max-width: 1280px) 300', $attribute->offsetGet('sizes'));
          break;

        case 2:
        case 5:
        case 8:
          $this->assertEquals('500', $attribute->offsetGet('sizes'));
          break;

        case 9:
          $this->assertNull($attribute->offsetGet('sizes'));
          break;

        default:
          $this->error('Un-expected content detected inside <picture> element.');
          break;
      }

      // Test attribute 'src' is correct for fallback img tag.
      if ($position == 9) {
        $this->assertStringContainsString('/medium-80/', $attribute->offsetGet('src'));
      }
      // Test attribute has correct 'srcset'.
      else {
        $srcset = $attribute->offsetGet('srcset');
        $srcsets = explode(',', $srcset);
        array_walk($srcsets, function (&$src) {
          [$key, $multiplier] = explode(' ', $src);
          $src = [
            'url' => $key,
            'multiplier' => $multiplier,
          ];
        });
        $this->assertEquals('1x', $srcsets[0]['multiplier']);
        $this->assertEquals('2x', $srcsets[1]['multiplier']);
        switch ($position) {
          case 0:
            $this->assertStringContainsString('/100w100h-avif-80/', $srcsets[0]['url']);
            $this->assertStringContainsString('/200w200h-avif-40/', $srcsets[1]['url']);
            break;

          case 1:
            $this->assertStringContainsString('/300w300h-avif-80/', $srcsets[0]['url']);
            $this->assertStringContainsString('/600w600h-avif-40/', $srcsets[1]['url']);
            break;

          case 2:
            $this->assertStringContainsString('/500w500h-avif-80/', $srcsets[0]['url']);
            $this->assertStringContainsString('/1000w1000h-avif-40/', $srcsets[1]['url']);
            break;

          case 3:
            $this->assertStringContainsString('/100w100h-webp-80/', $srcsets[0]['url']);
            $this->assertStringContainsString('/200w200h-webp-40/', $srcsets[1]['url']);
            break;

          case 4:
            $this->assertStringContainsString('/300w300h-webp-80/', $srcsets[0]['url']);
            $this->assertStringContainsString('/600w600h-webp-40/', $srcsets[1]['url']);
            break;

          case 5:
            $this->assertStringContainsString('/500w500h-webp-80/', $srcsets[0]['url']);
            $this->assertStringContainsString('/1000w1000h-webp-40/', $srcsets[1]['url']);
            break;

          case 6:
            $this->assertStringContainsString('/100w100h-jpeg-80/', $srcsets[0]['url']);
            $this->assertStringContainsString('/200w200h-jpeg-40/', $srcsets[1]['url']);
            break;

          case 7:
            $this->assertStringContainsString('/300w300h-jpeg-80/', $srcsets[0]['url']);
            $this->assertStringContainsString('/600w600h-jpeg-40/', $srcsets[1]['url']);
            break;

          case 8:
            $this->assertStringContainsString('/500w500h-jpeg-80/', $srcsets[0]['url']);
            $this->assertStringContainsString('/1000w1000h-jpeg-40/', $srcsets[1]['url']);
            break;

          default:
            $this->error('Un-expected content detected inside <picture> element.');
            break;
        }
      }
    }
  }

  /**
   * Test error handling for invalid media.
   */
  public function testInvalidMediaHandling() {
    $service = \Drupal::service('rift.picture');
    $config = $this->getBaseConfig();
    
    // Test with null media
    $elements = $service->responsivePicture(null, $config);
    /** @var \Drupal\Core\Render\Renderer $renderer */
    $renderer = \Drupal::service('renderer');
    $string = $renderer->renderInIsolation($elements);
    $this->assertEquals('<picture ></picture>', $string);
    
    // Note: Testing with invalid media object types is not possible due to strict type hints
    // The method signature requires ?MediaInterface, so passing stdClass would cause a TypeError
  }

  /**
   * Test error handling for missing builder manager.
   */
  public function testMissingBuilderManager() {
    $media = $this->media;
    $service = \Drupal::service('rift.picture');
    $config = $this->getBaseConfig();
    
    // Test with invalid configuration that might cause builder manager issues
    $config['invalid_setting'] = 'should_not_break';
    $elements = $service->responsivePicture($media, $config);
    
    // Should still return a valid response or empty array, not throw exception
    $this->assertIsArray($elements);
  }

  /**
   * Get default configuration to be used for testing.
   */
  protected function getBaseConfig() {
    return [
      'sizes' => 'md:100 lg:300 xl:500',
      'aspect_ratios' => '1x1 1x1 1x1',
      'screens' => [
        'xs' => [
          'width' => 640,
          'media' => '(max-width: 640px)',
        ],
        'sm' => [
          'width' => 768,
          'media' => '(max-width: 768px)',
        ],
        'md' => [
          'width' => 1024,
          'media' => '(max-width: 1024px)',
        ],
        'lg' => [
          'width' => 1280,
          'media' => '(max-width: 1280px)',
        ],
        'xl' => [
          'width' => 1536,
          'media' => '',
        ],
      ],
      'transforms' => '',
      'quality' => [
        '1x' => 80,
        '2x' => 40,
      ],
      'formats' => [
        'avif',
        'webp',
        'jpeg',
      ],
      'attributes' => [
        'loading' => 'lazy',
        'preload' => '',
      ],
      'fallback_transform' => 'medium-80',
      'url_generation_strategy' => 'combined_image_style',
      'multipliers' => [
        '1x',
        '2x',
      ],
    ];
  }

} 