<?php

namespace Drupal\rift\ExistingSite;

use Drupal\file\Entity\File;
use Drupal\Tests\TestFileCreationTrait;
use weitzman\DrupalTestTraits\Entity\MediaCreationTrait;
use weitzman\DrupalTestTraits\ExistingSiteBase;

/**
 * Tests for Rift Picture URL generation strategies.
 */
class RiftPictureUrlGeneratorTest extends ExistingSiteBase {
  use MediaCreationTrait;
  use TestFileCreationTrait;

  /**
   * The sample image File entity to embed.
   *
   * @var \Drupal\media\MediaInterface
   */
  protected $media;

  /**
   * The sample image File entity to embed.
   *
   * @var \Drupal\file\FileInterface
   */
  protected $file;

  /**
   * URI of the image file being used for testing.
   *
   * @var string
   */
  protected $imageFileUri;

  /**
   * {@inheritdoc}
   */
  protected function setUp() : void {
    parent::setUp();
    $this->imageFileUri = $this->getTestFiles('image')[0]->uri;
    // Create a sample host entity to embed images in.
    $this->file = File::create([
      'uri' => $this->imageFileUri,
    ]);
    $this->file
      ->save();
    $this->markEntityForCleanup($this->file);
    $this->media = $this->createMedia(
      [
        'bundle' => 'image',
        'name' => 'test_name_test_name_test_name',
        'field_media_image' => [
          'target_id' => $this->file->id(),
          'alt' => 'test_alt_test_alt_test_alt',
          'title' => 'test_title_test_title_test_title',
        ],
      ]
    );
    $this->markEntityForCleanup($this->media);
  }

  /**
   * Test responsive picture 'placeholdit' plugin.
   */
  public function testPlaceholditImageGenerator() {
    $media = $this->media;
    $service = \Drupal::service('rift.picture');
    $config = $this->getBaseConfig();
    $config['url_generation_strategy'] = 'placeholdit';
    $elements = $service->responsivePicture($media, $config);
    $this->assertEquals(10, count($elements['#context']['content']));
    $map = [
      0 => [
        'https://place-hold.it/100x100?text=100w100h-avif-80',
        'https://place-hold.it/200x200?text=200w200h-avif-40',
      ],
      1 => [
        'https://place-hold.it/300x300?text=300w300h-avif-80',
        'https://place-hold.it/600x600?text=600w600h-avif-40',
      ],
      2 => [
        'https://place-hold.it/500x500?text=500w500h-avif-80',
        'https://place-hold.it/1000x1000?text=1000w1000h-avif-40',
      ],
      3 => [
        'https://place-hold.it/100x100?text=100w100h-webp-80',
        'https://place-hold.it/200x200?text=200w200h-webp-40',
      ],
      4 => [
        'https://place-hold.it/300x300?text=300w300h-webp-80',
        'https://place-hold.it/600x600?text=600w600h-webp-40',
      ],
      5 => [
        'https://place-hold.it/500x500?text=500w500h-webp-80',
        'https://place-hold.it/1000x1000?text=1000w1000h-webp-40',
      ],
      6 => [
        'https://place-hold.it/100x100?text=100w100h-jpeg-80',
        'https://place-hold.it/200x200?text=200w200h-jpeg-40',
      ],
      7 => [
        'https://place-hold.it/300x300?text=300w300h-jpeg-80',
        'https://place-hold.it/600x600?text=600w600h-jpeg-40',
      ],
      8 => [
        'https://place-hold.it/500x500?text=500w500h-jpeg-80',
        'https://place-hold.it/1000x1000?text=1000w1000h-jpeg-40',
      ],
    ];
    foreach ($elements["#context"]["content"] as $ndx => $content) {
      /** @var \Drupal\Core\Template\Attribute $attributes */
      $attribute = $content['#context']['attributes'];
      if ($content['#context']['tag_name'] == "source") {
        $srcset_string = (string) $attribute->offsetGet('srcset');
        $srcset_items = explode(',', $srcset_string);
        $this->assertEquals(2, count($srcset_items));
        [$first_url] = explode(' ', $srcset_items[0]);
        [$second_url] = explode(' ', $srcset_items[1]);
        $this->assertEquals($map[$ndx][0], $first_url);
        $this->assertEquals($map[$ndx][1], $second_url);
      }
    }
  }

  /**
   * Test responsive picture 'dummyimage' plugin.
   */
  public function testDummyImage() {
    $media = $this->media;
    $service = \Drupal::service('rift.picture');
    $config = $this->getBaseConfig();
    $config['url_generation_strategy'] = 'dummyimage';
    $elements = $service->responsivePicture($media, $config);
    $this->assertEquals(10, count($elements['#context']['content']));
    $map = [
      0 => [
        'https://dummyimage.com/100x100/eee/222?text=100w100h-avif-80',
        'https://dummyimage.com/200x200/eee/222?text=200w200h-avif-40',
      ],
      1 => [
        'https://dummyimage.com/300x300/eee/222?text=300w300h-avif-80',
        'https://dummyimage.com/600x600/eee/222?text=600w600h-avif-40',
      ],
      2 => [
        'https://dummyimage.com/500x500/eee/222?text=500w500h-avif-80',
        'https://dummyimage.com/1000x1000/eee/222?text=1000w1000h-avif-40',
      ],
      3 => [
        'https://dummyimage.com/100x100/eee/222?text=100w100h-webp-80',
        'https://dummyimage.com/200x200/eee/222?text=200w200h-webp-40',
      ],
      4 => [
        'https://dummyimage.com/300x300/eee/222?text=300w300h-webp-80',
        'https://dummyimage.com/600x600/eee/222?text=600w600h-webp-40',
      ],
      5 => [
        'https://dummyimage.com/500x500/eee/222?text=500w500h-webp-80',
        'https://dummyimage.com/1000x1000/eee/222?text=1000w1000h-webp-40',
      ],
      6 => [
        'https://dummyimage.com/100x100/eee/222?text=100w100h-jpeg-80',
        'https://dummyimage.com/200x200/eee/222?text=200w200h-jpeg-40',
      ],
      7 => [
        'https://dummyimage.com/300x300/eee/222?text=300w300h-jpeg-80',
        'https://dummyimage.com/600x600/eee/222?text=600w600h-jpeg-40',
      ],
      8 => [
        'https://dummyimage.com/500x500/eee/222?text=500w500h-jpeg-80',
        'https://dummyimage.com/1000x1000/eee/222?text=1000w1000h-jpeg-40',
      ],
    ];
    foreach ($elements["#context"]["content"] as $ndx => $content) {
      /** @var \Drupal\Core\Template\Attribute $attributes */
      $attribute = $content['#context']['attributes'];
      if ($content['#context']['tag_name'] == "source") {
        $srcset_string = (string) $attribute->offsetGet('srcset');
        $srcset_items = explode(',', $srcset_string);
        $this->assertEquals(2, count($srcset_items));
        [$first_url] = explode(' ', $srcset_items[0]);
        [$second_url] = explode(' ', $srcset_items[1]);
        $this->assertEquals($map[$ndx][0], $first_url);
        $this->assertEquals($map[$ndx][1], $second_url);
      }
    }
  }

  /**
   * Test responsive picture 'combined_image_style' plugin.
   */
  public function testCombinedImageStyle() {
    $media = $this->media;
    $service = \Drupal::service('rift.picture');
    $config = $this->getBaseConfig();
    $config['url_generation_strategy'] = 'combined_image_style';
    $elements = $service->responsivePicture($media, $config);
    
    // Test that we get a valid response
    $this->assertNotEmpty($elements);
    
    // Test that the response contains expected structure
    if (isset($elements['#context']['content'])) {
      $this->assertEquals(10, count($elements['#context']['content']));
      
      foreach ($elements["#context"]["content"] as $content) {
        /** @var \Drupal\Core\Template\Attribute $attributes */
        $attribute = $content['#context']['attributes'];
        if ($content['#context']['tag_name'] == "source") {
          $srcset_string = (string) $attribute->offsetGet('srcset');
          $srcset_items = explode(',', $srcset_string);
          $this->assertEquals(2, count($srcset_items));
          
          // Test that URLs contain expected patterns for combined image style
          [$first_url] = explode(' ', $srcset_items[0]);
          [$second_url] = explode(' ', $srcset_items[1]);
          
          // URLs should contain image style patterns
          $this->assertStringContainsString('/sites/default/files/styles/', $first_url);
          $this->assertStringContainsString('/sites/default/files/styles/', $second_url);
        }
      }
    }
  }

  /**
   * Get default configuration to be used for testing.
   */
  protected function getBaseConfig() {
    return [
      'sizes' => 'md:100 lg:300 xl:500',
      'aspect_ratios' => '1x1 1x1 1x1',
      'screens' => [
        'xs' => [
          'width' => 640,
          'media' => '(max-width: 640px)',
        ],
        'sm' => [
          'width' => 768,
          'media' => '(max-width: 768px)',
        ],
        'md' => [
          'width' => 1024,
          'media' => '(max-width: 1024px)',
        ],
        'lg' => [
          'width' => 1280,
          'media' => '(max-width: 1280px)',
        ],
        'xl' => [
          'width' => 1536,
          'media' => '',
        ],
      ],
      'transforms' => '',
      'quality' => [
        '1x' => 80,
        '2x' => 40,
      ],
      'formats' => [
        'avif',
        'webp',
        'jpeg',
      ],
      'attributes' => [
        'loading' => 'lazy',
        'preload' => '',
      ],
      'fallback_transform' => 'medium-80',
      'url_generation_strategy' => 'combined_image_style',
      'multipliers' => [
        '1x',
        '2x',
      ],
    ];
  }

} 