<?php

namespace Drupal\rijksvideo\EventSubscriber;

use Drupal\file\FileInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\file\FileRepositoryInterface;
use Drupal\rijksvideo\Event\RijksvideoEvent;
use Drupal\rijksvideo\Service\RijksvideoEmbedParser;
use GuzzleHttp\ClientInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Enrich the creation of a new media entity.
 */
class RijksvideoEventSubscriber implements EventSubscriberInterface {

  use StringTranslationTrait;

  /**
   * Constructs a new RijksvideoEventSubscriber object.
   *
   * @param \Drupal\file\FileRepositoryInterface $fileRepository
   *   The file repository service.
   * @param \GuzzleHttp\ClientInterface $httpClient
   *   The HTTP client service.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   * @param \Drupal\rijksvideo\Service\RijksvideoEmbedParser $embedParser
   *   The embed parser service.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $stringTranslation
   *   The string translation service.
   */
  public function __construct(
    protected FileRepositoryInterface $fileRepository,
    protected ClientInterface $httpClient,
    protected LoggerChannelFactoryInterface $loggerFactory,
    protected RijksvideoEmbedParser $embedParser,
    protected MessengerInterface $messenger,
    TranslationInterface $stringTranslation,
  ) {
    $this->stringTranslation = $stringTranslation;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events = [];
    $events[RijksvideoEvent::RIJKSVIDEO_MEDIA_PRESAVE][] = ['fillMetaData'];
    return $events;
  }

  /**
   * Enrich the creation of a new media entity.
   *
   * Supports dual-source: prioritizes embed code (v2), falls back to XML (v1).
   *
   * @param \Drupal\rijksvideo\Event\RijksvideoEvent $event
   *   The Rijksvideo event.
   *
   * @throws \Drupal\Core\Entity\EntityStorageException
   */
  public function fillMetaData(RijksvideoEvent $event) {
    /** @var \Drupal\media\MediaInterface $entity */
    $entity = $event->getEntity();

    if ($entity->bundle() !== "rijksvideo") {
      return;
    }

    // Priority 1: Check for embed code (v2 method).
    if ($this->hasEmbedCode($entity)) {
      $this->fillFromEmbedCode($entity);
      $event->setEntity($entity);
      return;
    }

    // Priority 2: Fallback to XML file (v1 backwards compatibility).
    if ($this->hasXmlFile($entity)) {
      $this->fillFromXmlFile($entity);
      $event->setEntity($entity);
      return;
    }

    // No valid source found.
    $this->loggerFactory->get('rijksvideo')
      ->warning('Rijksvideo media item has neither embed code nor XML file.');
  }

  /**
   * Check if entity has embed code.
   *
   * @param \Drupal\media\MediaInterface $entity
   *   The media entity.
   *
   * @return bool
   *   TRUE if embed code exists and is not empty.
   */
  protected function hasEmbedCode($entity): bool {
    return $entity->hasField('field_rijksvideo_embed') &&
           !$entity->get('field_rijksvideo_embed')->isEmpty();
  }

  /**
   * Check if entity has XML file.
   *
   * @param \Drupal\media\MediaInterface $entity
   *   The media entity.
   *
   * @return bool
   *   TRUE if XML file exists and is not empty.
   */
  protected function hasXmlFile($entity): bool {
    return $entity->hasField('field_rijksvideo_file') &&
           !$entity->get('field_rijksvideo_file')->isEmpty();
  }

  /**
   * Fill metadata from embed code (v2 method).
   *
   * @param \Drupal\media\MediaInterface $entity
   *   The media entity.
   */
  protected function fillFromEmbedCode($entity) {
    $embedCode = $entity->get('field_rijksvideo_embed')->value;

    // Parse embed code.
    $parsed = $this->embedParser->parseEmbedCode($embedCode);

    if (empty($parsed['video_url'])) {
      $this->loggerFactory->get('rijksvideo')->error('Could not parse embed code: @embed', ['@embed' => $embedCode]);
      return;
    }

    // Store basic metadata.
    $this->storeBasicMetadata($entity, $parsed);

    // Process thumbnail.
    $this->processThumbnail($entity, $parsed);

    // Process caption.
    $this->processCaption($entity, $parsed);

    // Process auto-detected files.
    $this->processAutoDetectedFiles($entity, $parsed);

    // Check for missing files and warn user.
    $this->checkForMissingFiles($parsed);
  }

  /**
   * Fill metadata from XML file (v1 backwards compatibility).
   *
   * @param \Drupal\media\MediaInterface $entity
   *   The media entity.
   */
  protected function fillFromXmlFile($entity) {
    /** @var \Drupal\file\FileInterface $xmlFile */
    $xmlFile = $entity->get('field_rijksvideo_file')->entity;

    if (!$xmlFile) {
      return;
    }

    $xmlPath = $xmlFile->getFileUri();
    $xmlContent = file_get_contents($xmlPath);

    if (empty($xmlContent)) {
      $this->loggerFactory->get('rijksvideo')->error('Could not read XML file: @path', ['@path' => $xmlPath]);
      return;
    }

    try {
      $xml = simplexml_load_string($xmlContent);

      if ($xml === FALSE) {
        $this->loggerFactory->get('rijksvideo')->error('Invalid XML structure in file: @path', ['@path' => $xmlPath]);
        return;
      }

      // Parse XML and extract metadata.
      $parsed = $this->parseXmlContent($xml);

      // Store metadata using same methods as embed code.
      $this->storeBasicMetadata($entity, $parsed);
      $this->processThumbnail($entity, $parsed);
      $this->processCaption($entity, $parsed);

    }
    catch (\Exception $e) {
      $this->loggerFactory->get('rijksvideo')->error('XML parsing error: @error', ['@error' => $e->getMessage()]);
    }
  }

  /**
   * Parse XML content to extract metadata.
   *
   * @param \SimpleXMLElement $xml
   *   The parsed XML object.
   *
   * @return array
   *   Parsed metadata array compatible with embed code format.
   */
  protected function parseXmlContent(\SimpleXMLElement $xml): array {
    $parsed = [
      'video_url' => '',
      'poster_url' => '',
      'caption_url' => '',
      'caption_label' => '',
      'detected_urls' => [],
    ];

    // Extract video URL from sources.
    if (isset($xml->sources->source)) {
      foreach ($xml->sources->source as $source) {
        $quality = (string) $source['quality'];
        $url = trim((string) $source);

        if ($quality === 'web_hd' || $quality === 'hd') {
          $parsed['video_url'] = $url;
          break;
        }
        elseif (empty($parsed['video_url'])) {
          $parsed['video_url'] = $url;
        }
      }
    }

    // Extract poster/thumbnail URL.
    if (isset($xml->thumbnail)) {
      $parsed['poster_url'] = trim((string) $xml->thumbnail);
    }

    // Extract caption/subtitle URL.
    if (isset($xml->captions->caption)) {
      $parsed['caption_url'] = trim((string) $xml->captions->caption);
      $parsed['caption_label'] = isset($xml->captions->caption['label']) ?
        (string) $xml->captions->caption['label'] : 'Subtitles';
    }

    // Extract audio description URL.
    if (isset($xml->audio)) {
      $parsed['detected_urls']['audio_url'] = trim((string) $xml->audio);
    }

    // Extract transcription.
    if (isset($xml->transcription)) {
      $parsed['detected_urls']['transcription_url'] = trim((string) $xml->transcription);
    }

    return $parsed;
  }

  /**
   * Store basic metadata from parsed embed code.
   *
   * @param \Drupal\media\MediaInterface $entity
   *   The media entity.
   * @param array $parsed
   *   Parsed embed code data.
   */
  protected function storeBasicMetadata($entity, array $parsed) {
    // Store extracted URLs.
    if ($entity->hasField('field_rijksvideo_video_url')) {
      $entity->set('field_rijksvideo_video_url', $parsed['video_url']);
    }

    if ($entity->hasField('field_rijksvideo_poster_url') && !empty($parsed['poster_url'])) {
      $entity->set('field_rijksvideo_poster_url', $parsed['poster_url']);
    }

    // Fill sources field like original module.
    if ($entity->hasField('field_rijksvideo_sources') && !empty($parsed['video_url'])) {
      $sources = [
        'web_hd' => [
          'url' => $parsed['video_url'],
          'type' => 'MP4',
          'size' => $this->getVideoSize($parsed),
        ],
      ];
      $entity->set('field_rijksvideo_sources', json_encode($sources));
    }

    // Auto-generate title if empty.
    if (empty($entity->label()) || $entity->label() === 'Temporary title') {
      $title = $this->embedParser->generateTitleFromVideoUrl($parsed['video_url']);
      if (!empty($title)) {
        $entity->setName($title);
      }
    }
  }

  /**
   * Get file size from external video URL.
   *
   * @param array $parsed
   *   Parsed embed code data.
   *
   * @return string
   *   The formatted file size string.
   */
  protected function getVideoSize($parsed) {
    $size = '';
    $headers = get_headers($parsed['video_url'], 1);
    if (isset($headers['Content-Length'])) {
      $size_bytes = is_array($headers['Content-Length']) ? end($headers['Content-Length']) : $headers['Content-Length'];
      $size = $this->formatBytes((int) $size_bytes);
    }

    return $size;
  }

  /**
   * Process and download thumbnail.
   *
   * @param \Drupal\media\MediaInterface $entity
   *   The media entity.
   * @param array $parsed
   *   Parsed embed code data.
   */
  protected function processThumbnail($entity, array $parsed) {
    if (empty($parsed['poster_url']) || !$entity->hasField('field_rijksvideo_thumbnail')) {
      return;
    }

    try {
      // Delete old thumbnail file if it exists.
      if (!$entity->get('field_rijksvideo_thumbnail')->isEmpty()) {
        /** @var \Drupal\file\FileInterface $oldThumbnail */
        $oldThumbnail = $entity->get('field_rijksvideo_thumbnail')->entity;
        if ($oldThumbnail) {
          $oldThumbnail->delete();
        }
      }

      // Download new thumbnail.
      $thumbnail = $this->downloadRemoteFile($parsed['poster_url'], $entity->label(), 'thumbnail');
      if ($thumbnail) {
        $entity->set('field_rijksvideo_thumbnail', $thumbnail);
        $entity->set('thumbnail', $thumbnail);
      }
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('rijksvideo')->warning('Could not download thumbnail: @url - @error', [
        '@url' => $parsed['poster_url'],
        '@error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Process and download caption file.
   *
   * @param \Drupal\media\MediaInterface $entity
   *   The media entity.
   * @param array $parsed
   *   Parsed embed code data.
   */
  protected function processCaption($entity, array $parsed) {
    if (empty($parsed['caption_url']) || !$entity->hasField('field_rijksvideo_caption')) {
      return;
    }

    try {
      // Delete old caption file if it exists.
      if (!$entity->get('field_rijksvideo_caption')->isEmpty()) {
        /** @var \Drupal\file\FileInterface $oldCaption */
        $oldCaption = $entity->get('field_rijksvideo_caption')->entity;
        if ($oldCaption) {
          $oldCaption->delete();
        }
      }

      // Download new caption.
      $caption = $this->downloadRemoteFile($parsed['caption_url'], $entity->label(), 'caption');
      if ($caption) {
        $entity->set('field_rijksvideo_caption', $caption);
      }
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('rijksvideo')->warning('Could not download caption: @url - @error', [
        '@url' => $parsed['caption_url'],
        '@error' => $e->getMessage(),
      ]);
    }
  }

  /**
   * Check for missing files in parsed data and show warnings to user.
   *
   * @param array $parsed
   *   Parsed embed code or XML data.
   */
  protected function checkForMissingFiles(array $parsed) {
    $unavailableUrls = [];

    // Check detected URLs if they exist and are downloadable.
    if (isset($parsed['detected_urls'])) {
      if (!$this->checkForMissingAudioFile($parsed)) {
        $unavailableUrls['audio_url'] = $parsed['detected_urls']['audio_url'];
      }
      if (!$this->checkForMissingTranscriptionFile($parsed)) {
        $unavailableUrls['transcription_url'] = $parsed['detected_urls']['transcription_url'];
      }
      if (!$this->checkForMissingCaptionFile($parsed)) {
        $unavailableUrls['caption_url'] = $parsed['caption_url'];
      }
      // Show warning messages for unavailable URLs.
      if (!empty($unavailableUrls)) {
        foreach ($unavailableUrls as $type => $unavailableUrl) {
          $this->messenger->addWarning(
            $this->t('Could not find @resource. Please check if there is a link for @type and add it manually.', [
              '@resource' => $unavailableUrl,
              '@type' => $type,
            ])
          );
        }
      }
    }
  }

  /**
   * Check if the audio file is missing.
   *
   * @param array $parsed
   *   Parsed embed code or XML data.
   *
   * @return bool
   *   FALSE if the audio file is missing, TRUE otherwise.
   */
  protected function checkForMissingAudioFile(array $parsed) {
    $detected = $parsed['detected_urls'];
    if (!empty($detected['audio_url'])) {
      if (!$this->checkUrlExists($detected['audio_url'])) {
        return FALSE;
      }
    }
    return TRUE;
  }

  /**
   * Check if the transcription file is missing.
   *
   * @param array $parsed
   *   Parsed embed code or XML data.
   *
   * @return bool
   *   FALSE if the transcription file is missing, TRUE otherwise.
   */
  protected function checkForMissingTranscriptionFile(array $parsed) {
    $detected = $parsed['detected_urls'];
    if (!empty($detected['transcription_url'])) {
      if (!$this->checkUrlExists($detected['transcription_url'])) {
        return FALSE;
      }
    }
    return TRUE;
  }

  /**
   * Check if the caption file is missing.
   *
   * @param array $parsed
   *   Parsed embed code or XML data.
   *
   * @return bool
   *   FALSE if the caption file is missing, TRUE otherwise.
   */
  protected function checkForMissingCaptionFile(array $parsed) {
    if (!empty($parsed['caption_url'])) {
      if (!$this->checkUrlExists($parsed['caption_url'])) {
        return FALSE;
      }
    }
    return TRUE;
  }

  /**
   * Process auto-detected files (audio, transcription, caption VTT).
   *
   * @param \Drupal\media\MediaInterface $entity
   *   The media entity.
   * @param array $parsed
   *   Parsed embed code data.
   */
  protected function processAutoDetectedFiles($entity, array $parsed) {
    if (empty($parsed['detected_urls'])) {
      return;
    }

    $detected = $parsed['detected_urls'];

    $this->processAudioDescription($entity, $detected);
    $this->processTranscription($entity, $detected);
    $this->processAutoCaptionFile($entity, $detected, $parsed);
  }

  /**
   * Process audio description URL.
   *
   * @param \Drupal\media\MediaInterface $entity
   *   The media entity.
   * @param array $detected
   *   Detected URLs array.
   */
  protected function processAudioDescription($entity, array $detected) {
    if (empty($detected['audio_url']) || !$entity->hasField('field_rijksvideo_audio')) {
      return;
    }

    if ($this->checkUrlExists($detected['audio_url'])) {
      $entity->set('field_rijksvideo_audio', $detected['audio_url']);
    }
  }

  /**
   * Process transcription text file.
   *
   * @param \Drupal\media\MediaInterface $entity
   *   The media entity.
   * @param array $detected
   *   Detected URLs array.
   */
  protected function processTranscription($entity, array $detected) {
    if (empty($detected['transcription_url']) || !$entity->hasField('field_rijksvideo_transcription')) {
      return;
    }

    try {
      $transcriptionText = $this->fetchTextFile($detected['transcription_url']);
      if (!empty($transcriptionText)) {
        $entity->set('field_rijksvideo_transcription', [
          'value' => $transcriptionText,
          'format' => 'plain_text',
        ]);
        $entity->set('field_rijksvideo_trans_url', rijksvideo__normalize_url($detected['transcription_url']));
      }
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('rijksvideo')->notice('Could not fetch transcription: @url', [
        '@url' => $detected['transcription_url'],
      ]);
    }
  }

  /**
   * Process auto-detected caption VTT file.
   *
   * @param \Drupal\media\MediaInterface $entity
   *   The media entity.
   * @param array $detected
   *   Detected URLs array.
   * @param array $parsed
   *   Parsed embed code data.
   */
  protected function processAutoCaptionFile($entity, array $detected, array $parsed) {
    if (empty($detected['caption_vtt_url']) || !$entity->hasField('field_rijksvideo_caption')) {
      return;
    }

    // Only download if caption field is empty and no caption was in embed code.
    if (!$entity->get('field_rijksvideo_caption')->isEmpty() || !empty($parsed['caption_url'])) {
      return;
    }

    try {
      $caption = $this->downloadRemoteFile($detected['caption_vtt_url'], $entity->label(), 'caption');
      if ($caption) {
        $entity->set('field_rijksvideo_caption', $caption);
      }
    }
    catch (\Exception $e) {
      // Silent fail - this is auto-detection, not critical.
    }
  }

  /**
   * Download remote file and save it locally.
   *
   * @param string $url
   *   The URL to download.
   * @param string $title
   *   The media title (for filename generation).
   * @param string $type
   *   File type (thumbnail, subtitle, etc.).
   *
   * @return \Drupal\file\FileInterface|null
   *   The saved file entity or NULL on failure.
   */
  protected function downloadRemoteFile(string $url, string $title, string $type): ?FileInterface {
    if (empty($url)) {
      return NULL;
    }

    // Ensure URL has protocol.
    if (!preg_match('/^https?:\/\//', $url)) {
      $url = 'https://' . ltrim($url, '/');
    }

    try {
      // Get file content.
      $response = $this->httpClient->request('GET', $url);
      $data = (string) $response->getBody();

      // Generate filename.
      $extension = pathinfo(parse_url($url, PHP_URL_PATH), PATHINFO_EXTENSION);
      $sanitizedTitle = $this->sanitizeFilename($title);
      $filename = 'rv_' . $sanitizedTitle . '_' . $type . '.' . $extension;
      $destination = 'public://' . $filename;

      // Save file (1 = FILE_EXISTS_REPLACE in Drupal 10)
      return $this->fileRepository->writeData($data, $destination, 1);
    }
    catch (\Exception $e) {
      $this->loggerFactory->get('rijksvideo')->error('Could not download file: @url - @error', [
        '@url' => $url,
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Sanitize filename.
   *
   * @param string $filename
   *   The filename that needs to be sanitized.
   *
   * @return string
   *   The sanitized filename.
   */
  protected function sanitizeFilename(string $filename): string {
    // Remove HTML entities.
    $filename = html_entity_decode($filename, ENT_QUOTES | ENT_HTML5, 'UTF-8');
    // Replace non-alphanumeric characters with underscore.
    $filename = preg_replace('/[^a-zA-Z0-9_]+/', '_', $filename);
    // Convert to lowercase.
    $filename = strtolower($filename);
    // Remove leading/trailing underscores.
    $filename = trim($filename, '_');
    // Limit length.
    $filename = substr($filename, 0, 50);

    return $filename;
  }

  /**
   * Check if a URL exists and is accessible.
   *
   * @param string $url
   *   The URL to check.
   *
   * @return bool
   *   TRUE if URL exists and is accessible, FALSE otherwise.
   */
  protected function checkUrlExists(string $url): bool {
    if (empty($url)) {
      return FALSE;
    }

    try {
      $response = $this->httpClient->request('HEAD', $url);
      return $response->getStatusCode() === 200;
    }
    catch (\Exception $e) {
      return FALSE;
    }
  }

  /**
   * Fetch text content from a remote file.
   *
   * @param string $url
   *   The URL of the text file.
   *
   * @return string
   *   The text content, or empty string on failure.
   */
  protected function fetchTextFile(string $url): string {
    if (empty($url)) {
      return '';
    }

    try {
      $response = $this->httpClient->request('GET', $url);
      if ($response->getStatusCode() === 200) {
        return (string) $response->getBody();
      }
      return '';
    }
    catch (\Exception $e) {
      return '';
    }
  }

  /**
   * Format bytes to string.
   *
   * @param int $bytes
   *   The number of bytes.
   * @param int $precision
   *   The precision.
   *
   * @return string
   *   The formatted string of bytes.
   */
  protected function formatBytes(int $bytes, $precision = 2): string {
    $units = ['B', 'KB', 'MB', 'GB', 'TB'];

    $bytes = max($bytes, 0);
    $pow = floor(($bytes ? log($bytes) : 0) / log(1024));
    $pow = min($pow, count($units) - 1);

    $bytes /= pow(1024, $pow);

    return round($bytes, $precision) . ' ' . $units[$pow];
  }

}
