<?php

namespace Drupal\rijksvideo\Plugin\media\Source;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\media\Attribute\MediaSource;
use Drupal\media\MediaInterface;
use Drupal\media\MediaSourceBase;
use Drupal\media\MediaTypeInterface;

/**
 * Rijksvideo embed code media source.
 *
 * Provides a media source for videos from the Rijksvideo platform,
 * using iframe embed codes as the source.
 */
#[MediaSource(
  id: "rijksvideo_embed",
  label: new TranslatableMarkup("Rijksvideo Embed"),
  description: new TranslatableMarkup("Use iframe embed codes from rovid.nl video platform."),
  allowed_field_types: ["string", "string_long"],
  default_thumbnail_filename: "video.png",
  forms: [
    "media_library_add" => "Drupal\\rijksvideo\\Form\\RijksvideoEmbedForm",
  ],
)]
class RijksvideoEmbed extends MediaSourceBase {

  /**
   * Key for "Video URL" metadata attribute.
   */
  const METADATA_ATTRIBUTE_VIDEO_URL = 'video_url';

  /**
   * Key for "Poster URL" metadata attribute.
   */
  const METADATA_ATTRIBUTE_POSTER_URL = 'poster_url';

  /**
   * Key for "Audio URL" metadata attribute.
   */
  const METADATA_ATTRIBUTE_AUDIO_URL = 'audio_url';

  /**
   * {@inheritdoc}
   */
  public function getMetadataAttributes() {
    return [
      static::METADATA_ATTRIBUTE_VIDEO_URL => $this->t('Video URL'),
      static::METADATA_ATTRIBUTE_POSTER_URL => $this->t('Poster URL'),
      static::METADATA_ATTRIBUTE_AUDIO_URL => $this->t('Audio description URL'),
    ];
  }

  /**
   * {@inheritdoc}
   *
   * Supports dual-source: embed code (v2) or XML file (v1 backwards compat).
   */
  public function getMetadata(MediaInterface $media, $attribute_name) {
    // Try stored fields first (works for both v1 and v2).
    $fieldValue = $this->getMetadataFromStoredFields($media, $attribute_name);
    if ($fieldValue !== NULL) {
      return $fieldValue;
    }

    // Fallback: Try parsing embed code if available (v2 method).
    return $this->getMetadataFromEmbedCode($media, $attribute_name);
  }

  /**
   * Get metadata from already-stored fields.
   *
   * @param \Drupal\media\MediaInterface $media
   *   The media entity.
   * @param string $attribute_name
   *   The attribute name.
   *
   * @return mixed|null
   *   The field value or NULL if not found.
   */
  protected function getMetadataFromStoredFields(MediaInterface $media, $attribute_name) {
    switch ($attribute_name) {
      case static::METADATA_ATTRIBUTE_VIDEO_URL:
        return $this->getFieldValue($media, 'field_rijksvideo_video_url');

      case static::METADATA_ATTRIBUTE_POSTER_URL:
        return $this->getFieldValue($media, 'field_rijksvideo_poster_url');

      case static::METADATA_ATTRIBUTE_AUDIO_URL:
        return $this->getFieldValue($media, 'field_rijksvideo_audio');

      case 'thumbnail_uri':
        return $this->getThumbnailUri($media);

      case 'default_name':
        return $this->getDefaultName($media);
    }

    return NULL;
  }

  /**
   * Get metadata from embed code parsing.
   *
   * @param \Drupal\media\MediaInterface $media
   *   The media entity.
   * @param string $attribute_name
   *   The attribute name.
   *
   * @return mixed|null
   *   The parsed value or parent metadata.
   */
  protected function getMetadataFromEmbedCode(MediaInterface $media, $attribute_name) {
    $embed_code = $media->get($this->configuration['source_field'])->value;

    if (empty($embed_code)) {
      return parent::getMetadata($media, $attribute_name);
    }

    /** @var \Drupal\rijksvideo\Service\RijksvideoEmbedParser $parser */
    $parser = \Drupal::service('rijksvideo.embed_parser');
    $parsed = $parser->parseEmbedCode($embed_code);

    switch ($attribute_name) {
      case static::METADATA_ATTRIBUTE_VIDEO_URL:
        return $parsed['video_url'] ?? NULL;

      case static::METADATA_ATTRIBUTE_POSTER_URL:
        return $parsed['poster_url'] ?? NULL;

      case static::METADATA_ATTRIBUTE_AUDIO_URL:
        return $parsed['detected_urls']['audio_url'] ?? NULL;
    }

    return parent::getMetadata($media, $attribute_name);
  }

  /**
   * Get field value helper.
   *
   * @param \Drupal\media\MediaInterface $media
   *   The media entity.
   * @param string $field_name
   *   The field name.
   *
   * @return string|null
   *   The field value or NULL.
   */
  protected function getFieldValue(MediaInterface $media, string $field_name): ?string {
    if ($media->hasField($field_name) && !$media->get($field_name)->isEmpty()) {
      return $media->get($field_name)->value;
    }
    return NULL;
  }

  /**
   * Get thumbnail URI.
   *
   * @param \Drupal\media\MediaInterface $media
   *   The media entity.
   *
   * @return string|null
   *   The thumbnail URI or parent metadata.
   */
  protected function getThumbnailUri(MediaInterface $media) {
    if ($media->hasField('field_rijksvideo_thumbnail') &&
        !$media->get('field_rijksvideo_thumbnail')->isEmpty()) {
      /** @var \Drupal\file\FileInterface $thumbnail */
      $thumbnail = $media->get('field_rijksvideo_thumbnail')->entity;
      if ($thumbnail) {
        return $thumbnail->getFileUri();
      }
    }
    return parent::getMetadata($media, 'thumbnail_uri');
  }

  /**
   * Get default name from video URL.
   *
   * @param \Drupal\media\MediaInterface $media
   *   The media entity.
   *
   * @return string|null
   *   The generated title or parent metadata.
   */
  protected function getDefaultName(MediaInterface $media) {
    $videoUrl = $this->getFieldValue($media, 'field_rijksvideo_video_url');

    if (!empty($videoUrl)) {
      /** @var \Drupal\rijksvideo\Service\RijksvideoEmbedParser $parser */
      $parser = \Drupal::service('rijksvideo.embed_parser');
      $title = $parser->generateTitleFromVideoUrl($videoUrl);
      if ($title) {
        return $title;
      }
    }

    return parent::getMetadata($media, 'default_name');
  }

  /**
   * {@inheritdoc}
   */
  public function createSourceField(MediaTypeInterface $type) {
    return parent::createSourceField($type)
      ->set('label', 'Embed Code')
      ->set('description', 'Paste the complete iframe embed code from the beeldbank.')
      ->set('required', TRUE);
  }

}
