<?php

namespace Drupal\rijksvideo\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\media\MediaInterface;

/**
 * Preprocessor service for Rijksvideo templates.
 *
 * Handles template preprocessing logic for rijksvideo media entities.
 */
class RijksvideoPreprocessor {

  /**
   * Constructs a RijksvideoPreprocessor object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $fileUrlGenerator
   *   The file URL generator service.
   * @param \Drupal\rijksvideo\Service\RijksvideoUtility $utility
   *   The Rijksvideo utility service.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected FileUrlGeneratorInterface $fileUrlGenerator,
    protected RijksvideoUtility $utility,
  ) {}

  /**
   * Preprocess variables for rijksvideo media templates.
   *
   * @param array<string, mixed> $variables
   *   The variables array from the preprocess hook.
   */
  public function preprocessMedia(array &$variables): void {
    /** @var \Drupal\media\MediaInterface $media */
    $media = $variables['elements']['#media'];

    if ($media->bundle() !== 'rijksvideo') {
      return;
    }

    $this->processAudioDescription($variables, $media);
    $this->processThumbnail($variables, $media);
    $this->processCaption($variables, $media);
    $this->processTranscription($variables, $media);
    $this->processSources($variables, $media);
    $this->processVideoUrl($variables, $media);
    $this->processSettings($variables);
  }

  /**
   * Process audio description URL.
   *
   * @param array<string, mixed> $variables
   *   The variables array.
   * @param \Drupal\media\MediaInterface $media
   *   The media entity.
   */
  protected function processAudioDescription(array &$variables, MediaInterface $media): void {
    $audio_url = $media->get('field_rijksvideo_audio')->value;
    if ($audio_url) {
      $variables['audio'] = $this->utility->normalizeUrl($audio_url);
    }
  }

  /**
   * Process thumbnail/poster image.
   *
   * @param array<string, mixed> $variables
   *   The variables array.
   * @param \Drupal\media\MediaInterface $media
   *   The media entity.
   */
  protected function processThumbnail(array &$variables, MediaInterface $media): void {
    $thumbnail = $media->get('field_rijksvideo_thumbnail')->entity;
    if ($thumbnail) {
      /** @var \Drupal\file\FileInterface $thumbnail */
      $variables['poster'] = $this->fileUrlGenerator->generateAbsoluteString(
        $thumbnail->getFileUri()
      );
    }
  }

  /**
   * Process caption/subtitles file.
   *
   * @param array<string, mixed> $variables
   *   The variables array.
   * @param \Drupal\media\MediaInterface $media
   *   The media entity.
   */
  protected function processCaption(array &$variables, MediaInterface $media): void {
    $caption = $media->get('field_rijksvideo_caption')->entity;
    if ($caption) {
      /** @var \Drupal\file\FileInterface $caption */
      $variables['caption'] = $this->fileUrlGenerator->generateAbsoluteString(
        $caption->getFileUri()
      );
    }
  }

  /**
   * Process transcription field.
   *
   * @param array<string, mixed> $variables
   *   The variables array.
   * @param \Drupal\media\MediaInterface $media
   *   The media entity.
   */
  protected function processTranscription(array &$variables, MediaInterface $media): void {
    if (!$media->get('field_rijksvideo_transcription')->isEmpty()) {
      $variables['transcription'] = $media->get('field_rijksvideo_transcription')
        ->first()
        ->view();
    }
  }

  /**
   * Process video sources.
   *
   * @param array<string, mixed> $variables
   *   The variables array.
   * @param \Drupal\media\MediaInterface $media
   *   The media entity.
   */
  protected function processSources(array &$variables, MediaInterface $media): void {
    $sources_string = $media->get('field_rijksvideo_sources')->value;
    $sources = $sources_string ? json_decode($sources_string, TRUE) : [];

    foreach ($sources as $key => $source) {
      // For some reason URLs don't always contain https URLs or even http.
      if (!empty($source['url'])) {
        $sources[$key]['url'] = $this->utility->normalizeUrl($source['url']);
      }
    }
    $variables['sources'] = $sources;
  }

  /**
   * Process video URL (prioritize web_hd, then web_mp4, then fallback).
   *
   * @param array<string, mixed> $variables
   *   The variables array.
   * @param \Drupal\media\MediaInterface $media
   *   The media entity.
   */
  protected function processVideoUrl(array &$variables, MediaInterface $media): void {
    $sources = $variables['sources'] ?? [];

    if (!empty($sources['web_hd']['url'])) {
      $variables['video_url'] = $sources['web_hd']['url'];
    }
    elseif (!empty($sources['web_mp4']['url'])) {
      $variables['video_url'] = $sources['web_mp4']['url'];
    }
    elseif (!$media->get('field_rijksvideo_video_url')->isEmpty()) {
      // Fallback to direct video_url field for backwards compatibility.
      $variables['video_url'] = $media->get('field_rijksvideo_video_url')->value;
    }
  }

  /**
   * Process module settings.
   *
   * @param array<string, mixed> $variables
   *   The variables array.
   */
  protected function processSettings(array &$variables): void {
    $config = $this->configFactory->get('rijksvideo.settings');
    $variables['only_show_download_accordion'] = $config->get('only_show_download_accordion');
    $variables['#cache']['tags'][] = 'config:rijksvideo.settings';
  }

}
