<?php

namespace Drupal\rijksvideo\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\media_library\Form\AddFormBase;
use Drupal\media_library\MediaLibraryUiBuilder;
use Drupal\media_library\OpenerResolverInterface;
use Drupal\rijksvideo\Service\RijksvideoEmbedParser;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Creates a form to create Rijksvideo media entities from embed codes.
 *
 * @internal
 *   Form classes are internal.
 */
class RijksvideoEmbedForm extends AddFormBase {

  /**
   * Constructs a new RijksvideoEmbedForm.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\media_library\MediaLibraryUiBuilder $library_ui_builder
   *   The media library UI builder.
   * @param \Drupal\rijksvideo\Service\RijksvideoEmbedParser $embedParser
   *   The Rijksvideo embed parser service.
   * @param \Drupal\media_library\OpenerResolverInterface|null $opener_resolver
   *   The opener resolver.
   */
  public function __construct(
    EntityTypeManagerInterface $entity_type_manager,
    MediaLibraryUiBuilder $library_ui_builder,
    protected RijksvideoEmbedParser $embedParser,
    ?OpenerResolverInterface $opener_resolver = NULL,
  ) {
    parent::__construct($entity_type_manager, $library_ui_builder, $opener_resolver);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('media_library.ui_builder'),
      $container->get('rijksvideo.embed_parser'),
      $container->get('media_library.opener_resolver'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return $this->getBaseFormId() . '_rijksvideo_embed';
  }

  /**
   * {@inheritdoc}
   */
  protected function getMediaType(FormStateInterface $form_state) {
    if ($this->mediaType) {
      return $this->mediaType;
    }

    $media_type = parent::getMediaType($form_state);
    if ($media_type->getSource()->getPluginId() !== 'rijksvideo_embed') {
      throw new \InvalidArgumentException('Can only add media types which use the rijksvideo_embed source plugin.');
    }
    return $media_type;
  }

  /**
   * {@inheritdoc}
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array<string, mixed>
   *   The form array.
   */
  protected function buildInputElement(array $form, FormStateInterface $form_state): array {
    $media_type = $this->getMediaType($form_state);

    // Add a container to group the input elements for styling purposes.
    $form['container'] = [
      '#type' => 'container',
    ];

    $form['container']['embed_code'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Add @type via embed code', [
        '@type' => $media_type->label(),
      ]),
      '#description' => $this->t('Paste the complete iframe embed code from the beeldbank. Click "Kopieer Embed code naar klembord" in the beeldbank and paste here.'),
      '#required' => TRUE,
      '#rows' => 3,
      '#attributes' => [
        'placeholder' => $this->t('<iframe src="...">'),
      ],
    ];

    $form['container']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add'),
      '#button_type' => 'primary',
      '#validate' => ['::validateEmbedCode'],
      '#submit' => ['::addButtonSubmit'],
      '#ajax' => [
        'callback' => '::updateFormCallback',
        'wrapper' => 'media-library-wrapper',
        // Add a fixed URL to post the form since AJAX forms are automatically
        // posted to <current> instead of $form['#action'].
        'url' => Url::fromRoute('media_library.ui'),
        'options' => [
          'query' => $this->getMediaLibraryState($form_state)->all() + [
            FormBuilderInterface::AJAX_FORM_REQUEST => TRUE,
          ],
        ],
      ],
    ];
    return $form;
  }

  /**
   * Validates the embed code.
   *
   * @param array<string, mixed> $form
   *   The complete form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   */
  public function validateEmbedCode(array &$form, FormStateInterface $form_state): void {
    $embed_code = $form_state->getValue('embed_code');
    if ($embed_code) {
      try {
        $parsed = $this->embedParser->parseEmbedCode($embed_code);
        if (empty($parsed['video_url'])) {
          $form_state->setErrorByName('embed_code', $this->t('Could not extract video URL from embed code. Please check that you pasted the complete iframe embed code.'));
        }
      }
      catch (\Exception $e) {
        $form_state->setErrorByName('embed_code', $this->t('Invalid embed code: @message', ['@message' => $e->getMessage()]));
      }
    }
  }

  /**
   * Submit handler for the add button.
   *
   * @param array<string, mixed> $form
   *   The form render array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function addButtonSubmit(array $form, FormStateInterface $form_state): void {
    $this->processInputValues([$form_state->getValue('embed_code')], $form, $form_state);
  }

}
