<?php

namespace Drupal\rijksvideo\Service;

use Drupal\Core\Form\FormStateInterface;

/**
 * Utility service for Rijksvideo module.
 *
 * Provides common helper methods used across the module.
 */
class RijksvideoUtility {

  /**
   * Normalize URL to ensure https protocol.
   *
   * @param string $url
   *   The URL to be normalized.
   *
   * @return string
   *   Returns the normalized URL with https protocol.
   */
  public function normalizeUrl(string $url): string {
    $url = str_replace(['http://', 'https://'], '', $url);
    $url = 'https://' . $url;

    return trim($url);
  }

  /**
   * Checks if the name is auto-generated (UUID-like pattern).
   *
   * @param string $name
   *   The media name to check.
   *
   * @return bool
   *   TRUE if the name appears to be auto-generated.
   */
  public function isAutoGeneratedName(string $name): bool {
    // Check for pattern like "media:rijksvideo:uuid".
    return preg_match('/^media:rijksvideo:[a-f0-9-]{36}$/i', $name) === 1;
  }

  /**
   * Sets a placeholder name when no name is submitted.
   *
   * Use as form validation callback.
   *
   * @param array<string, mixed> $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state interface.
   */
  public function setEmptyName(array &$form, FormStateInterface $form_state): void {
    if ($form_state->isSubmitted()) {
      $name = $form_state->getValue('name');

      if (empty($name) || empty($name[0]['value'])) {
        $form_state->setValue('name', [
          ['value' => 'Temporary title'],
        ]);
      }
    }
  }

}
