#!/bin/bash
set -e

echo "Attempting to update Rivet icons..."

# Fetch the latest release tag from GitHub API
LATEST_TAG=$(curl -s https://api.github.com/repos/indiana-university/rivet-icons/releases/latest | grep '"tag_name":' | cut -d '"' -f 4)

if [ -z "$LATEST_TAG" ]; then
  echo "Error: Failed to fetch the latest tag from GitHub."
  exit 1
fi

# Remove 'v' prefix from tag for directory name, e.g., v3.0.1 -> 3.0.1
VERSION_NO_V=$(echo "$LATEST_TAG" | sed 's/^v//')

echo "Latest Rivet icons version tag: $LATEST_TAG (version: $VERSION_NO_V)"

# Create a temporary directory for download and extraction
# cspell:disable-next-line
TEMP_DIR=$(mktemp -d)
echo "Created temporary directory: $TEMP_DIR"

# Ensure temporary directory is cleaned up on script exit
trap 'echo "Cleaning up temporary directory $TEMP_DIR..."; rm -rf "$TEMP_DIR"' EXIT

ARCHIVE_URL="https://github.com/indiana-university/rivet-icons/archive/refs/tags/${LATEST_TAG}.tar.gz"
ARCHIVE_FILE="$TEMP_DIR/icons.tar.gz"

echo "Downloading Rivet icons archive from $ARCHIVE_URL..."
curl -sL "$ARCHIVE_URL" -o "$ARCHIVE_FILE"

echo "Extracting icons from archive $ARCHIVE_FILE..."
tar -xzf "$ARCHIVE_FILE" -C "$TEMP_DIR"

# Define paths for source and target directories
EXTRACTED_CONTENTS_DIR="$TEMP_DIR/rivet-icons-${VERSION_NO_V}"
SOURCE_ICONS_DIR="$EXTRACTED_CONTENTS_DIR/src/icons"
TARGET_ICON_DIR="images/rivet-icons" # Relative to project root

# Check if the expected source icons directory exists
if [ ! -d "$SOURCE_ICONS_DIR" ]; then
  echo "Error: Source icons directory '$SOURCE_ICONS_DIR' not found after extraction."
  echo "Contents of $EXTRACTED_CONTENTS_DIR:"
  ls -A "$EXTRACTED_CONTENTS_DIR" # List contents to help debug if path is wrong
  exit 1
fi

echo "Recreating target directory: $TARGET_ICON_DIR to ensure a clean state..."
rm -rf "$TARGET_ICON_DIR"
mkdir -p "$TARGET_ICON_DIR"

echo "Copying icons from $SOURCE_ICONS_DIR to $TARGET_ICON_DIR..."
# Check if source directory has content before attempting to copy
if [ -n "$(ls -A "$SOURCE_ICONS_DIR")" ]; then
    cp -a "$SOURCE_ICONS_DIR"/* "$TARGET_ICON_DIR/"
else
    echo "Warning: Source icons directory '$SOURCE_ICONS_DIR' is empty. No icons were copied."
fi

echo "Attempting to copy LICENSE file from package root to $TARGET_ICON_DIR/LICENSE.txt..."
TARGET_LICENSE_FILE_PATH="$TARGET_ICON_DIR/LICENSE.txt"
LICENSE_COPIED_FLAG=false
# Define potential source license file names in order of preference
SOURCE_LICENSE_CANDIDATES=(
    "$EXTRACTED_CONTENTS_DIR/LICENSE"
    "$EXTRACTED_CONTENTS_DIR/LICENSE.md"
    "$EXTRACTED_CONTENTS_DIR/LICENSE.txt"
    "$EXTRACTED_CONTENTS_DIR/LICENSE.TXT"
)

for candidate_path in "${SOURCE_LICENSE_CANDIDATES[@]}"; do
    if [ -f "$candidate_path" ]; then
        echo "Found license file at '$candidate_path'. Copying to '$TARGET_LICENSE_FILE_PATH'..."
        cp "$candidate_path" "$TARGET_LICENSE_FILE_PATH"
        LICENSE_COPIED_FLAG=true
        break # Stop after finding the first suitable candidate
    fi
done

if [ "$LICENSE_COPIED_FLAG" = false ]; then
    echo "Warning: No suitable LICENSE file (LICENSE, LICENSE.md, LICENSE.txt, LICENSE.TXT) found in package root '$EXTRACTED_CONTENTS_DIR'."
fi

echo "Updating SVG attributes to ensure aria-hidden=\"true\" and focusable=\"false\"..."
find "$TARGET_ICON_DIR" -name '*.svg' -print0 | while IFS= read -r -d $'\0' svg_file; do

  # Ensure focusable="false"
  # Re-grep because file may have changed after aria-hidden modification
  if ! grep -q '<svg[^>]*focusable="false"[^>]*>' "$svg_file"; then
    if grep -q '<svg[^>]*focusable=' "$svg_file"; then
      # echo "    Updating focusable attribute in $svg_file" # Optional: for verbose logging
      sed -i '' 's/\(<svg[^>]*\)focusable="[^"]*"/\1focusable="false"/' "$svg_file"
    else
      # echo "    Adding focusable=\"false\" to $svg_file" # Optional: for verbose logging
      sed -i '' 's/<svg/<svg focusable="false"/' "$svg_file"
    fi
  #else
    # echo "    focusable=\"false\" already present or correct in $svg_file" # Optional: for verbose logging
  fi

  # Ensure aria-hidden="true"
  if ! grep -q '<svg[^>]*aria-hidden="true"[^>]*>' "$svg_file"; then
    if grep -q '<svg[^>]*aria-hidden=' "$svg_file"; then
      # echo "    Updating aria-hidden attribute in $svg_file" # Optional: for verbose logging
      sed -i '' 's/\(<svg[^>]*\)aria-hidden="[^"]*"/\1aria-hidden="true"/' "$svg_file"
    else
      # echo "    Adding aria-hidden=\"true\" to $svg_file" # Optional: for verbose logging
      sed -i '' 's/<svg/<svg aria-hidden="true"/' "$svg_file"
    fi
  #else
    # echo "    aria-hidden=\"true\" already present or correct in $svg_file" # Optional: for verbose logging
  fi

done

echo "Rivet icons updated successfully to version $LATEST_TAG in $TARGET_ICON_DIR."
echo "Done."
