<?php

namespace Drupal\robots_rerouter\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\ResponseHeaderBag;

/**
 * Returns a dynamic robots.txt response based on the environment.
 */
class Robots extends ControllerBase {

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * Constructs a new Robots controller.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory service.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    FileSystemInterface $file_system
  ) {
    // configFactory is inherited from ControllerBase
    $this->configFactory = $config_factory;
    $this->fileSystem = $file_system;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('file_system')
    );
  }

  /**
   * Returns the contents of the correct robots.txt file based on environment.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The robots.txt response.
   */
  public function generate(): Response {
    $current_url = Url::fromRoute('<front>', [], ['absolute' => TRUE])
      ->toString();
    $current_host = str_replace(['https://', 'http://'], '', $current_url);
    $current_host = rtrim($current_host, '/');

    $config = $this->configFactory->get('robots_rerouter.config');
    $production_host = $config->get('production_site');
    $production_file = 'public://' . ltrim($config->get('production_file'), '/');
    $fallback_file = 'public://' . ltrim($config->get('fallback_file'), '/');

    $file_uri = ($current_host === $production_host)
      ? $production_file
      : $fallback_file;

    // Ensure directory exists.
    $this->fileSystem->prepareDirectory(
      dirname($file_uri),
      FileSystemInterface::CREATE_DIRECTORY | FileSystemInterface::MODIFY_PERMISSIONS
    );

    // Try to resolve file path.
    $real_path = $this->fileSystem->realpath($file_uri);

    // If file is missing, create fallback content.
    if (!$real_path || !file_exists($real_path)) {
      file_put_contents($this->fileSystem->realpath($file_uri), "# robots.txt not found\nUser-agent: *\nDisallow: /");
      $real_path = $this->fileSystem->realpath($file_uri);
    }

    $contents = file_get_contents($real_path);

    return new Response($contents, Response::HTTP_OK, [
      'Content-Type' => 'text/plain',
      'Content-Disposition' => ResponseHeaderBag::DISPOSITION_INLINE,
    ]);
  }

}
