<?php

namespace Drupal\robots_rerouter\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Configuration form for Robots Rerouter settings.
 */
class ConfigForm extends ConfigFormBase {

  /**
   * The current request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a new ConfigForm instance.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack service.
   */
  public function __construct(RequestStack $request_stack) {
    $this->requestStack = $request_stack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('request_stack')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['robots_rerouter.config'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'robots_rerouter_config_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $request = $this->requestStack->getCurrentRequest();
    $current_host = $request->getHost();

    $config = $this->config('robots_rerouter.config');

    $form['production_site'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Production Site Hostname'),
      '#description' => $this->t('Only this hostname will serve the official robots.txt. Example: www.example.com'),
      '#maxlength' => 128,
      '#size' => 64,
      '#default_value' => $config->get('production_site'),
      '#required' => TRUE,
    ];

    $form['production_file'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Path to Production robots.txt'),
      '#description' => $this->t('<strong>Do not include public://</strong> Enter a path relative to the public file system (e.g., robots/good_robots.txt).'),
      '#default_value' => $config->get('production_file'),
      '#required' => TRUE,
    ];

    $form['fallback_file'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Path to Fallback (Disallow) robots.txt'),
      '#description' => $this->t('File path to use on non-production environments. <br /> <strong>Do not include public://</strong> Enter a path relative to the public file system (e.g., robots/nocrawl_robots.txt).'),
      '#default_value' => $config->get('fallback_file'),
      '#required' => TRUE,
    ];

    $form['current'] = [
      '#type' => 'markup',
      '#markup' => '<p><strong>Current Host:</strong> ' . $current_host . '</p>',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
   public function submitForm(array &$form, FormStateInterface $form_state): void {
     parent::submitForm($form, $form_state);

     // Sanitize file paths by removing any stream wrapper prefixes (e.g., public://).
     $sanitize = function ($path) {
       return preg_replace('/^[^:]+:\/\//', '', trim($path));
     };

     $this->config('robots_rerouter.config')
       ->set('production_site', $form_state->getValue('production_site'))
       ->set('production_file', $sanitize($form_state->getValue('production_file')))
       ->set('fallback_file', $sanitize($form_state->getValue('fallback_file')))
       ->save();
   }

}
