/**
 * Rocketship UI JS
 *
 * Contains: triggers for functions.
 * Functions themselves are split off and grouped below each behavior.
 */

(function (Drupal, window, document) {
  "use strict";

  // Set namespace for frontend UI javascript.
  if (typeof window.rocketshipUI === 'undefined') {
    window.rocketshipUI = {};
  }

  var self = window.rocketshipUI;

  ///////////////////////////////////////////////////////////////////////
  // Behavior for Tabs: triggers
  ///////////////////////////////////////////////////////////////////////

  Drupal.behaviors.rocketshipUILanguageMenu = {
    attach: (context, settings) => {
      const langNav = context.querySelectorAll('.block--language-blocklanguage-interface');

      if (langNav.length) {
        // Check the theme settings if it is set to 'dropdown'
        // located at: /admin/appearance/settings/theme_machine_name
        if (settings?.theme_settings?.language_dropdown === true || settings?.theme_settings?.language_dropdown === 1) {
          langNav.forEach((nav) => {
            self.languageDropdown(nav);
          });
        }
      }
    }
  };

  ///////////////////////////////////////////////////////////////////////
  // Behavior for Tabs: functions
  ///////////////////////////////////////////////////////////////////////

  /**
   * Dropdown menu.
   *
   * @param {HTMLElement} nav The navigation element.
   */
  self.languageDropdown = (nav) => {
    var activeLanguageHolder = nav.querySelector('.nav__active--language--dropdown');

    // Touch event to open/close.
    // Includes touch detection.
    activeLanguageHolder.addEventListener('touchstart', (e) => {
      self.touch = true;

      if (nav.classList.contains('js-open')) {
        self.navLangClose(nav);
      } else {
        self.navLangOpen(nav);
      }

      e.preventDefault();
    });

    // Reset the touch variable afterward.
    activeLanguageHolder.addEventListener('touchend', (e) => {
      // End
      setTimeout(() => {
        self.touch = false; // Reset bc we might be on a device that has mouse as well as touch capability
      }, 500); // Time it until after a 'click' would have fired on mobile (>300ms)

      e.preventDefault();
    });

    // Open/close on hover
    // If not in touch modus
    nav.addEventListener('mouseenter', (e) => {
      // If no touch triggered
      if (!self.touch) {
        self.navLangOpen(nav);
        e.preventDefault();
      }
    });

    // Close for normal menu
    // If not megamenu or small screen,
    nav.addEventListener('mouseleave', (e) => {
      self.navLangClose(nav);
      e.preventDefault();
    });

    // On window resize, reset the menu to closed state
    window.rocketshipUI.optimizedResize().add(() => {
      self.navLangClose(nav);
    });
  };

  /**
   * Opens the language navigation.
   *
   * @param {HTMLElement} target The target element to open.
   */
  self.navLangOpen = (target) => {
    target.classList.add('js-open');
  };

  /**
   * Closes the language navigation.
   *
   * @param {HTMLElement} target The target element to close.
   */
  self.navLangClose = (target) => {
    target.classList.remove('js-open');
  };

})(Drupal, window, document);
