// Concatenate & Minify JS
'use strict';

import gulp from 'gulp';
import sourcemaps from 'gulp-sourcemaps';
import concat from 'gulp-concat';
import flatten from 'gulp-flatten';
import notify from 'gulp-notify';
import pump from 'pump';
import { config } from '../config.mjs';
import { errorNotification } from './setup.mjs';

const options = {};

/**
 * Functions for compiling JS per part.
 */
const buildJsComponentsDev = () => {
  return gulp.src(config.js.components.src)
    .pipe(sourcemaps.init())
    .on('error', function (err) {
      errorNotification(this, err);
    })
    // Replace relative paths for files.
    .pipe(flatten({ includeParents: 0}))
    .on('error', function (err) {
      errorNotification(this, err);
    })
    .pipe(
      sourcemaps.write(config.js.dest, {
        sourceMappingURL: (file) => file.relative + '.map'
      })
    )
    // Replace relative paths for files.
    .pipe(flatten())
    .pipe(gulp.dest(config.js.dest))
};

const buildJsGlobalDev = () => {
  return pump([
    gulp.src(config.js.global.src),
    sourcemaps.init().on('error', function (err) {
      errorNotification(this, err);
    }),
    // Merge files into 1.
    concat('scripts.js', options).on('error', function (err) {
      errorNotification(this, err);
    }),
    sourcemaps.write('.', {
      sourceMappingURL: (file) => file.relative + '.map'
    }).on('error', function (err) {
      errorNotification(this, err);
    }),
    gulp.dest(config.js.dest)
      .on('error', function (err) {
        errorNotification(this, err);
      }),
  ]);
};

// Production
const buildJsComponentsProd = () => {
  return gulp.src(config.js.components.src)
    // replace relative paths for files
    .pipe(flatten({ includeParents: 0}))
    .on('error', function (err) {
      errorNotification(this, err);
    })
    // replace relative paths for files
    .pipe(flatten())
    .on('error', function (err) {
      errorNotification(this, err);
    })
    .pipe(gulp.dest(config.js.dest));
};

const buildJsGlobalProd = () => {
  return pump([
    gulp.src(config.js.global.src),
    // Merge files into 1.
    concat('scripts.js', options),
    gulp.dest(config.js.dest)
  ]);
};

/**
 * Components
 */

export const jsComponentsDev = buildJsComponentsDev;
export const jsComponentsProd = buildJsComponentsProd;


/**
 * Global
 */

export const jsGlobalDev = buildJsGlobalDev;
export const jsGlobalProd = buildJsGlobalProd;

/**
 * Combined tasks.
 */
const jsMessageDev = (done) => {
  notify(
    `
    ===========================================================================
    Don't commit development JS. Use 'gulp js:prod' instead.
    Sourcemap files won't work on environments and lead to errors & overhead.
    ===========================================================================`
  );
  done();
};

export const jsDev = gulp.series(jsMessageDev, jsGlobalDev, jsComponentsDev);
export const jsProd = gulp.series(jsGlobalProd, jsComponentsProd);

/**
 * Exports (optional, as direct exports are used above).
 */
export {
  buildJsGlobalDev,
  buildJsComponentsDev,
};
