<?php

/**
 * @file
 * Post update functions for the Rokka module.
 */

declare(strict_types=1);

use Drupal\Core\Config\Entity\ConfigEntityUpdater;
use Drupal\field\FieldStorageConfigInterface;
use Drupal\file\FileInterface;
use Drupal\media\MediaTypeInterface;
use Drupal\rokka\Entity\RokkaMetadataInterface;

/**
 * Add missing dependency on image style to each Rokka stack.
 */
function rokka_post_update_add_image_style_dependency(?array &$sandbox = NULL) {
  /** @var \Drupal\Core\Config\Entity\ConfigEntityUpdater $config_entity_updater */
  $config_entity_updater = \Drupal::classResolver(ConfigEntityUpdater::class);
  $config_entity_updater->update($sandbox, 'rokka_stack', function () {
    return TRUE;
  });
}

/**
 * Apply the dedicated media source to media types that contain Rokka videos.
 */
function rokka_post_update_apply_rokka_video_media_source() {
  $entity_type_manager = \Drupal::entityTypeManager();
  // Load all media types which use the 'video_file' media source and use Rokka
  // as a file storage.
  $media_types = $entity_type_manager
    ->getStorage('media_type')
    ->loadByProperties(['source' => 'video_file']);

  $media_types = array_filter($media_types, function (MediaTypeInterface $media_type) use ($entity_type_manager): bool {
    $source_field = $media_type->getSource()->getConfiguration()['source_field'] ?? NULL;
    if (!$source_field) {
      return FALSE;
    }
    $field_definitions = $entity_type_manager
      ->getStorage('field_storage_config')
      ->loadByProperties(['entity_type' => 'media', 'field_name' => $source_field]);
    $field_definition = reset($field_definitions);
    if (!$field_definition instanceof FieldStorageConfigInterface) {
      return FALSE;
    }
    $uri_scheme = $field_definition->getSetting('uri_scheme') ?? NULL;
    return $uri_scheme === 'rokka';
  });

  foreach ($media_types as $media_type) {
    $media_type->set('source', 'rokka_video');
    $media_type->save();
  }
}

/**
 * Generate missing thumbnails for Rokka videos.
 */
function rokka_post_update_generate_missing_thumbnails(?array &$sandbox = NULL) {
  $entity_type_manager = \Drupal::entityTypeManager();
  $file_storage = $entity_type_manager->getStorage('file');
  $media_storage = $entity_type_manager->getStorage('media');
  $metadata_storage = $entity_type_manager->getStorage('rokka_metadata');

  if (!isset($sandbox['media_ids'])) {
    $sandbox['media_ids'] = [];
    $sandbox['progress'] = 0;
    $sandbox['skipped'] = 0;
    $sandbox['total'] = 0;

    // Loop over the Rokka video media types and collect the media IDs.
    $media_types = $entity_type_manager
      ->getStorage('media_type')
      ->loadByProperties(['source' => 'rokka_video']);

    foreach ($media_types as $media_type) {
      $media_ids = $media_storage
        ->getQuery()
        ->accessCheck(FALSE)
        ->condition('bundle', $media_type->id())
        ->execute();
      $sandbox['media_ids'] = array_merge($sandbox['media_ids'], $media_ids);
    }
    $sandbox['total'] = count($sandbox['media_ids']);
  }

  // Small batches because generating thumbnails can be slow.
  $batch_size = 5;
  $media_ids = array_splice($sandbox['media_ids'], 0, $batch_size);
  $media_entities = $media_storage->loadMultiple($media_ids);
  foreach ($media_entities as $media_entity) {
    $sandbox['progress']++;

    /** @var \Drupal\rokka\Plugin\media\Source\RokkaVideo $source */
    $source = $media_entity->getSource();
    $uri = $source->getMetadata($media_entity, $source->getPluginDefinition()['thumbnail_uri_metadata_attribute']);
    if (empty($uri)) {
      $sandbox['skipped']++;
      continue;
    }

    $files = $file_storage->loadByProperties(['uri' => $uri]);
    $file = reset($files);
    $metadata = $metadata_storage->loadByUri($uri);

    if (!$file instanceof FileInterface || !$metadata instanceof RokkaMetadataInterface) {
      $sandbox['skipped']++;
      continue;
    }

    $media_entity->set('thumbnail', [
      'target_id' => $file->id(),
      'alt' => '',
      'width' => $metadata->getWidth(),
      'height' => $metadata->getHeight(),
    ])->save();
  }

  $sandbox['#finished'] = !empty($sandbox['media_ids']) ? $sandbox['progress'] / $sandbox['total'] : 1;

  return sprintf('Processed %d of %d thumbnails for Rokka videos. Skipped %d.', $sandbox['progress'], $sandbox['total'], $sandbox['skipped']);
}
