<?php

declare(strict_types=1);

namespace Drupal\rokka\Plugin\media\Source;

use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\media\Attribute\MediaSource;
use Drupal\media\MediaInterface;
use Drupal\media\Plugin\media\Source\VideoFile;
use Drupal\rokka\Entity\RokkaMetadataInterface;
use Drupal\rokka\RokkaMetadataStorageInterface;

/**
 * Media source for rokka video files.
 *
 * Provides thumbnail support for videos stored in Rokka.
 */
#[MediaSource(
  id: "rokka_video",
  label: new TranslatableMarkup("Rokka Video"),
  description: new TranslatableMarkup("Rokka videos with thumbnail support."),
  allowed_field_types: ["file"],
  default_thumbnail_filename: "video.jpg"
)]
class RokkaVideo extends VideoFile {

  use LoggerChannelTrait;

  /**
   * {@inheritdoc}
   */
  public function getMetadata(MediaInterface $media, $attribute_name) {
    if ($attribute_name === 'thumbnail_uri') {
      $videoUri = $media->get($this->configuration['source_field'])->entity?->getFileUri();
      if (empty($videoUri)) {
        $this->getLogger('rokka')->warning('Could not retrieve the file URI for the media entity with ID @id.', ['@id' => $media->id() ?? 'n/a']);
        return NULL;
      }

      $rokkaMetadataStorage = $this->entityTypeManager->getStorage('rokka_metadata');
      if (!$rokkaMetadataStorage instanceof RokkaMetadataStorageInterface) {
        $this->getLogger('rokka')->warning('Could not retrieve the Rokka metadata storage.');
        return NULL;
      }

      $videoMetadata = $rokkaMetadataStorage->loadByUri($videoUri);

      if (!$videoMetadata instanceof RokkaMetadataInterface) {
        $this->getLogger('rokka')->warning('Could not retrieve the Rokka metadata for the video with URI @uri.', ['@uri' => $videoUri]);
        return NULL;
      }

      $thumbnailMetadata = $rokkaMetadataStorage->getThumbnail($videoMetadata, 'jpg');

      if (!$thumbnailMetadata instanceof RokkaMetadataInterface) {
        $this->getLogger('rokka')->warning('Could not retrieve the Rokka metadata for the thumbnail of the video with URI @uri.', ['@uri' => $videoUri]);
        return NULL;
      }

      return $thumbnailMetadata->getUri();
    }

    return parent::getMetadata($media, $attribute_name);
  }

}
