<?php

declare(strict_types=1);

namespace Drupal\rokka;

use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Utility\Error;
use Drupal\media\MediaInterface;
use Drupal\rokka\Entity\RokkaMetadata;
use Drupal\rokka\Entity\RokkaStack;
use GuzzleHttp\Exception\GuzzleException;
use Psr\Log\LoggerInterface;
use Rokka\Client\Base;
use Rokka\Client\Factory;
use Rokka\Client\Image;
use Rokka\Client\TemplateHelper;
use Rokka\Client\User;

/**
 * Defines a RokkaService service.
 */
class RokkaService implements RokkaServiceInterface {

  /**
   * API Key.
   *
   * @var array|mixed|null
   */
  private $apiKey;

  /**
   * Organization name.
   *
   * @var array|mixed|null
   */
  private $organizationName;

  /**
   * API Endpoint.
   *
   * @var array|mixed|string|null
   */
  private $apiEndpoint;

  public function __construct(
    protected EntityTypeManagerInterface $entityTypeManager,
    ConfigFactory $configFactory,
    protected LoggerInterface $logger,
  ) {
    $config = $configFactory->get('rokka.settings');
    $this->apiKey = $config->get('api_key');
    $this->organizationName = $config->get('organization_name');
    $this->apiEndpoint = $config->get('api_endpoint') ?: Base::DEFAULT_API_BASE_URL;
  }

  /**
   * Returns the SEO compliant filename for the given image name.
   *
   * @param string $filename
   *   Filename.
   *
   * @return string
   *   Filename.
   */
  public static function cleanRokkaSeoFilename($filename): string {
    // Rokka.io accepts SEO URL part as "[a-z0-9-]" only, remove not valid
    // characters and replace them with '-'.
    return TemplateHelper::slugify($filename);
  }

  /**
   * {@inheritdoc}
   */
  public function getRokkaImageClient(): Image {
    return Factory::getImageClient($this->organizationName, $this->apiKey, '', $this->apiEndpoint);
  }

  /**
   * {@inheritdoc}
   */
  public function getRokkaUserClient(): User {
    return Factory::getUserClient($this->apiEndpoint);
  }

  /**
   * {@inheritdoc}
   */
  public function getRokkaOrganizationName() {
    return $this->organizationName;
  }

  /**
   * {@inheritdoc}
   */
  public function loadRokkaMetadataByUri(string $uri): array {
    // @todo Deprecate this method in favor of
    //   RokkaMetadataStorageInterface::loadByUri().
    return $this->entityTypeManager->getStorage('rokka_metadata')
      ->loadByProperties(['uri' => $uri]);
  }

  /**
   * Load Rokka metadata by binary hash.
   *
   * @param string $binary_hash
   *   Hash.
   *
   * @return \Drupal\Core\Entity\EntityInterface[]
   *   Entities.
   */
  public function loadRokkaMetadataByBinaryHash(string $binary_hash): array {
    return $this->entityTypeManager->getStorage('rokka_metadata')
      ->loadByProperties(['binary_hash' => $binary_hash]);
  }

  /**
   * Load Rokka metadata by media entity.
   *
   * @param \Drupal\media\MediaInterface $media
   *   Hash.
   *
   * @return \Drupal\rokka\Entity\RokkaMetadata|null
   *   Entities.
   */
  public function loadRokkaMetadataByMedia(MediaInterface $media): ?RokkaMetadata {
    $metadata = NULL;

    // Determine the field name of the image.
    $source_field_name = $media->bundle->entity->getSource()->getSourceFieldDefinition($media->bundle->entity)->getName();

    if ($media->{$source_field_name}->entity || !empty($media->{$source_field_name}->fids)) {
      // Load the Rokka metadata for the source image.
      $file = $media->{$source_field_name}->entity;
      // If the file was just uploaded, it's not yet saved, so we have to load
      // it from provided FileId.
      if (!$file) {
        $fid = reset($media->{$source_field_name}->fids);
        $file = $this->entityTypeManager->getStorage('file')->load($fid);
      }

      if ($file) {
        $metadata = $this->loadRokkaMetadataByUri($file->getFileUri());
        $metadata = reset($metadata);
      }
    }

    return $metadata ? $metadata : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function loadStackByName(string $name): ?RokkaStack {
    /** @var \Drupal\rokka\Entity\RokkaStack $stack */
    $stack = $this->entityTypeManager->getStorage('rokka_stack')->load($name);
    return $stack;
  }

  /**
   * {@inheritdoc}
   */
  public function countImagesWithHash(string $hash): int {
    return (int) $this->entityTypeManager->getStorage('rokka_metadata')
      ->getQuery()
      ->accessCheck(FALSE)
      ->condition('hash', $hash)
      ->count()
      ->execute();
  }

  /**
   * {@inheritdoc}
   */
  public function addAutoDescription(array $language_codes, string $image_hash): array {
    $client = $this->getRokkaImageClient();
    // cspell:disable-next-line
    $result = $client->addAutodescription($language_codes, $image_hash);

    if ($result->staticMetadata && !empty($result->staticMetadata['auto_description'])) {
      return $result->staticMetadata['auto_description']['descriptions'];
    }

    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function deleteSourceImage(string $hash): bool {
    $client = $this->getRokkaImageClient();
    try {
      return $client->deleteSourceImage($hash, $this->getRokkaOrganizationName());
    }
    catch (\Exception | GuzzleException $e) {
      Error::logException($this->logger, $e);
    }
    return FALSE;
  }

}
