<?php

declare(strict_types=1);

namespace Drupal\Tests\rokka\Kernel;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\rokka\Entity\RokkaMetadata;
use Drupal\Tests\rokka\Traits\RokkaMetadataCreationTrait;

/**
 * Tests the RokkaMetadata entity class.
 *
 * @group rokka
 * @coversDefaultClass \Drupal\rokka\Entity\RokkaMetadata
 */
class RokkaMetadataEntityTest extends EntityKernelTestBase {

  use RokkaMetadataCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['rokka'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('rokka_metadata');
  }

  /**
   * Tests getting and setting width and height.
   */
  public function testWidthAndHeight(): void {
    // Create a new RokkaMetadata entity.
    $metadata = $this->createRokkaMetadata();

    // Test that width and height are NULL when not set.
    $this->assertNull($metadata->getWidth());
    $this->assertNull($metadata->getHeight());

    // Test setting width and height.
    $metadata->setWidth(1920);
    $metadata->setHeight(1080);

    $this->assertEquals(1920, $metadata->getWidth());
    $this->assertEquals(1080, $metadata->getHeight());

    // Save the entity and verify the values persist.
    $metadata->save();

    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);
    $this->assertEquals(1920, $reloaded_metadata->getWidth());
    $this->assertEquals(1080, $reloaded_metadata->getHeight());
  }

  /**
   * Tests creating entity with custom field values.
   */
  public function testCreateWithCustomFieldValues(): void {
    $metadata = $this->createRokkaMetadata([
      'format' => 'png',
      'filesize' => 54321,
      'width' => 800,
      'height' => 600,
      'status' => FALSE,
    ]);

    $this->assertEquals(800, $metadata->getWidth());
    $this->assertEquals(600, $metadata->getHeight());
    $this->assertFalse($metadata->isPublished());
  }

  /**
   * Tests published status methods.
   */
  public function testPublishedStatus(): void {
    $metadata = $this->createRokkaMetadata();

    // By default, entities should be published.
    $this->assertTrue($metadata->isPublished());

    // Test unpublishing.
    $metadata->setUnpublished();
    $this->assertFalse($metadata->isPublished());

    // Test publishing.
    $metadata->setPublished();
    $this->assertTrue($metadata->isPublished());

    // Save and reload to verify persistence.
    $metadata->save();

    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);
    $this->assertTrue($reloaded_metadata->isPublished());

    // Verify that the unpublished state persists.
    $reloaded_metadata->setUnpublished();
    $reloaded_metadata->save();

    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);
    $this->assertFalse($reloaded_metadata->isPublished());
  }

  /**
   * Tests deprecated setPublished() parameter for backwards compatibility.
   *
   * @group legacy
   */
  public function testDeprecatedSetPublishedTrueParameter(): void {
    $this->expectDeprecation('Passing the $published argument to Drupal\rokka\Entity\RokkaMetadata::setPublished() is deprecated in rokka:3.1.0 and is removed from rokka:4.0.0. Use ::setUnpublished() instead. See https://www.drupal.org/node/3426517');

    $metadata = $this->createRokkaMetadata();
    $metadata->setPublished(TRUE);
    $this->assertTrue($metadata->isPublished());
  }

  /**
   * Tests deprecated setPublished() parameter for backwards compatibility.
   *
   * @group legacy
   */
  public function testDeprecatedSetPublishedFalseParameter(): void {
    $this->expectDeprecation('Passing the $published argument to Drupal\rokka\Entity\RokkaMetadata::setPublished() is deprecated in rokka:3.1.0 and is removed from rokka:4.0.0. Use ::setUnpublished() instead. See https://www.drupal.org/node/3426517');

    $metadata = $this->createRokkaMetadata();
    $metadata->setPublished(FALSE);
    $this->assertFalse($metadata->isPublished());
  }

}
