<?php

declare(strict_types=1);

namespace Drupal\rokka\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityPublishedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\Plugin\Field\FieldType\IntegerItem;
use Drupal\Core\TypedData\Exception\MissingDataException;
use Drupal\user\UserInterface;

/**
 * Defines the Rokka Metadata entity.
 *
 * @ingroup rokka
 *
 * @ContentEntityType(
 *   id = "rokka_metadata",
 *   label = @Translation("Rokka Metadata"),
 *   handlers = {
 *     "storage" = "Drupal\rokka\RokkaMetadataStorage",
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\rokka\Entity\Controller\RokkaMetadataListBuilder",
 *     "views_data" = "Drupal\rokka\Entity\RokkaMetadataViewsData",
 *     "storage_schema" = "Drupal\rokka\Entity\MetadataStorageSchema",
 *     "form" = {
 *       "default" = "Drupal\rokka\Form\RokkaMetadataForm",
 *       "add" = "Drupal\rokka\Form\RokkaMetadataForm",
 *       "edit" = "Drupal\rokka\Form\RokkaMetadataForm",
 *       "delete" = "Drupal\rokka\Form\RokkaMetadataDeleteForm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\rokka\RokkaMetadataHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "rokka_metadata",
 *   admin_permission = "administer rokka",
 *   entity_keys = {
 *     "binary_hash" = "binary_hash",
 *     "hash" = "hash",
 *     "id" = "id",
 *     "published" = "status",
 *     "status" = "status",
 *     "uid" = "user_id",
 *     "uri" = "uri",
 *     "uuid" = "uuid",
 *   },
 *   links = {
 *     "canonical" = "/admin/structure/rokka_metadata/{rokka_metadata}",
 *     "edit-form" = "/admin/structure/rokka_metadata/{rokka_metadata}/edit",
 *     "collection" = "/admin/structure/rokka_metadata",
 *   },
 *   field_ui_base_route = "rokka_metadata.settings",
 *
 * )
 */
class RokkaMetadata extends ContentEntityBase implements RokkaMetadataInterface {

  use EntityChangedTrait;
  use EntityPublishedTrait;

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage, array &$values): void {
    parent::preCreate($storage, $values);
    $values += [
      'user_id' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['user_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Authored by'))
      ->setDescription(t('The user ID of author of the Rokka Metadata entity.'))
      ->setRevisionable(FALSE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setTranslatable(FALSE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 5,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['hash'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Hash'))
      ->setDescription(t("The Rokka.io hash of the source object, based on metadata. Changes if metadata is modified, e.g. by 'Focal Point'."))
      ->setSettings([
        'max_length' => 40,
        'not null' => TRUE,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['binary_hash'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Hash'))
      ->setDescription(t('The Rokka.io hash of the binary image data of the source object.'))
      ->setSettings([
        'max_length' => 40,
        'not null' => TRUE,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['uri'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Uri'))
      ->setDescription(t('The original file URI.'))
      ->setSettings([
        'default_value' => '',
        'max_length' => 255,
        'not null' => TRUE,
      ]);

    $fields['filesize'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('File size'))
      ->setDescription(t('The original file size.'))
      ->setSettings([
        'size' => 'big',
        'not null' => TRUE,
        'unsigned' => TRUE,
      ]);

    $fields['height'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Height'))
      ->setDescription(t('The height of the image.'))
      ->setSettings([
        'size' => 'big',
        'not null' => TRUE,
        'unsigned' => TRUE,
      ]);

    $fields['width'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Width'))
      ->setDescription(t('The width of the image.'))
      ->setSettings([
        'size' => 'big',
        'not null' => TRUE,
        'unsigned' => TRUE,
      ]);

    $fields['format'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Image Format'))
      ->setDescription(t('The image format type of the file.'))
      ->setSettings([
        'default_value' => '',
        'max_length' => 255,
        'not null' => TRUE,
      ]);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Publishing status'))
      ->setDescription(t('A boolean indicating whether the Rokka Metadata is published.'))
      ->setDefaultValue(TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    $fields['static_metadata'] = BaseFieldDefinition::create('map')
      ->setLabel(t('Static metadata'))
      ->setDescription(t('Static metadata, like AI generated descriptions.'))
      ->setDefaultValue([])
      ->setRevisionable(FALSE)
      ->setTranslatable(FALSE);

    $fields['thumbnail'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Thumbnail'))
      ->setDescription(t('Whether this is metadata for a thumbnail.'))
      ->setDefaultValue(FALSE)
      ->setRevisionable(FALSE)
      ->setTranslatable(FALSE);

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function getHash(): string {
    return (string) $this->get('hash')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setHash(string $hash): RokkaMetadataInterface {
    $this->set('hash', $hash);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getBinaryHash(): string {
    return (string) $this->get('binary_hash')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setBinaryHash(string $binaryHash): RokkaMetadataInterface {
    $this->set('binary_hash', $binaryHash);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getFilesize(): int {
    return (int) $this->get('filesize')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setFilesize(int $filesize): RokkaMetadataInterface {
    $this->set('filesize', $filesize);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getUri(): string {
    return (string) $this->get('uri')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setUri(string $uri): RokkaMetadataInterface {
    $this->set('uri', $uri);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getHeight(): ?int {
    try {
      $item = $this->get('height')->first();
      if ($item instanceof IntegerItem && !$item->isEmpty()) {
        return (int) $item->value;
      }
    }
    catch (MissingDataException) {
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setHeight(int $height): RokkaMetadataInterface {
    $this->set('height', $height);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getWidth(): ?int {
    try {
      $item = $this->get('width')->first();
      if ($item instanceof IntegerItem && !$item->isEmpty()) {
        return (int) $item->value;
      }
    }
    catch (MissingDataException) {
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setWidth(int $width): RokkaMetadataInterface {
    $this->set('width', $width);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setStaticMetadata(array $metadata): RokkaMetadataInterface {
    $this->set('static_metadata', $metadata);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getStaticMetadata(): array {
    return $this->get('static_metadata')->getValue();
  }

  /**
   * {@inheritdoc}
   */
  public function getFormat(): string {
    return (string) $this->get('format')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setFormat(string $format): RokkaMetadataInterface {
    $this->set('format', $format);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime(): int {
    return (int) $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime(int $timestamp): RokkaMetadataInterface {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner(): UserInterface {
    $owner = $this->get('user_id')->entity;
    assert($owner instanceof UserInterface);
    return $owner;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId(): ?int {
    $id = $this->get('user_id')->target_id;
    if (is_numeric($id)) {
      return (int) $id;
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid): RokkaMetadataInterface {
    $this->set('user_id', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account): RokkaMetadataInterface {
    $this->set('user_id', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isThumbnail(): bool {
    return (bool) $this->get('thumbnail')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setThumbnail(bool $thumbnail): RokkaMetadataInterface {
    $this->set('thumbnail', $thumbnail);
    return $this;
  }

}
