<?php

declare(strict_types=1);

namespace Drupal\rokka;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Cache\MemoryCache\MemoryCacheInterface;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\Sql\SqlContentEntityStorage;
use Drupal\Core\ImageToolkit\ImageToolkitManager;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\file\Entity\File;
use Drupal\rokka\Entity\RokkaMetadataInterface;
use GuzzleHttp\ClientInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Mime\MimeTypeGuesserInterface;

/**
 * Storage handler for RokkaMetadata entities.
 */
class RokkaMetadataStorage extends SqlContentEntityStorage implements RokkaMetadataStorageInterface {

  use LoggerChannelTrait;

  public function __construct(
    EntityTypeInterface $entityType,
    Connection $database,
    EntityFieldManagerInterface $entityFieldManager,
    CacheBackendInterface $cache,
    LanguageManagerInterface $languageManager,
    MemoryCacheInterface $memoryCache,
    EntityTypeBundleInfoInterface $entityTypeBundleInfo,
    EntityTypeManagerInterface $entityTypeManager,
    protected RokkaServiceInterface $rokkaService,
    protected ImageToolkitManager $imageToolkitManager,
    protected ClientInterface $httpClient,
    protected MimeTypeGuesserInterface $mimeTypeGuesser,
  ) {
    parent::__construct($entityType, $database, $entityFieldManager, $cache, $languageManager, $memoryCache, $entityTypeBundleInfo, $entityTypeManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('database'),
      $container->get('entity_field.manager'),
      $container->get('cache.entity'),
      $container->get('language_manager'),
      $container->get('entity.memory_cache'),
      $container->get('entity_type.bundle.info'),
      $container->get('entity_type.manager'),
      $container->get('rokka.service'),
      $container->get('image.toolkit.manager'),
      $container->get('http_client'),
      $container->get('file.mime_type.guesser'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function loadByUri(string $uri): ?RokkaMetadataInterface {
    $entities = $this->loadByProperties(['uri' => $uri]);
    return $entities ? reset($entities) : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getThumbnail(RokkaMetadataInterface $videoMetadata, string $format = 'jpg'): ?RokkaMetadataInterface {
    $imageUri = preg_replace("/\.[^.]+$/", ".$format", $videoMetadata->getUri());
    $imageMetadata = $this->loadByUri($imageUri);

    // If the metadata for the thumbnail image does not exist yet, create it.
    if (!$imageMetadata instanceof RokkaMetadataInterface) {
      // The Rokka API does not return metadata about thumbnails, since they are
      // not considered to be "source images" with their own hash. We need to
      // retrieve the dimensions and file size by inspecting the image itself.
      $client = $this->rokkaService->getRokkaImageClient();
      $thumbnailUri = (string) $client->getSourceImageUri($videoMetadata->getHash(), 'dynamic', $format);
      if (empty($thumbnailUri)) {
        $this->getLogger('rokka')->warning('Could not retrieve the thumbnail URI for the Rokka image with hash @hash.', ['@hash' => $videoMetadata->getHash()]);
        return NULL;
      }

      $fileSize = $this->getFileSize($thumbnailUri);
      if (empty($fileSize)) {
        $this->getLogger('rokka')->warning('Could not retrieve the file size for the thumbnail URI @uri.', ['@uri' => $thumbnailUri]);
        return NULL;
      }

      [$width, $height] = $this->getImageDimensions($thumbnailUri);
      if (empty($width) || empty($height)) {
        $this->getLogger('rokka')->warning('Could not retrieve the dimensions for the thumbnail URI @uri.', ['@uri' => $thumbnailUri]);
        return NULL;
      }

      $imageMetadata = $videoMetadata->createDuplicate()
        ->setUri($imageUri)
        ->setFormat($format)
        ->setWidth($width)
        ->setHeight($height)
        ->setFileSize((int) $fileSize)
        ->setThumbnail(TRUE);
      $imageMetadata->save();
    }

    // Make sure a file entity exists for the thumbnail image. The media entity
    // expects this to be present.
    // @see \Drupal\media\Entity\Media::loadThumbnail()
    $file_storage = $this->entityTypeManager->getStorage('file');
    $files = $file_storage->loadByProperties(['uri' => $imageUri]);
    if (empty($files)) {
      $mimeType = $this->mimeTypeGuesser->guessMimeType($imageUri) ?: 'image/jpeg';

      $file = File::create(['uri' => $imageUri]);
      $file->setSize($imageMetadata->getFileSize());
      $file->setMimeType($mimeType);
      $file->setFilename(basename($imageUri));
      $file->setPermanent();

      if ($owner = $videoMetadata->getOwner()) {
        $file->setOwner($owner);
      }

      $file->save();
    }

    return $imageMetadata;
  }

  /**
   * Returns the dimensions of the image hosted at the given URI.
   *
   * @param string $uri
   *   The image URI.
   *
   * @return array
   *   A tuple containing width and height.
   */
  protected function getImageDimensions(string $uri): array {
    // We are using the GD toolkit because the Rokka toolkit will try to
    // retrieve the metadata from the RokkaMetadata entity which we are in the
    // process of creating.
    /** @var \Drupal\system\Plugin\ImageToolkit\GDToolkit $toolkit */
    $toolkit = $this->imageToolkitManager->createInstance('gd');
    $toolkit->setSource($uri);
    $toolkit->parseFile();

    return [$toolkit->getWidth(), $toolkit->getHeight()];
  }

  /**
   * Returns the file size of the file hosted at the given URI.
   *
   * @param string $uri
   *   The file URI.
   *
   * @return int
   *   The file size in bytes, or NULL if it could not be determined.
   */
  protected function getFileSize(string $uri): ?int {
    try {
      $response = $this->httpClient->head($uri);
      return (int) $response->getHeaderLine('Content-Length');
    }
    catch (\Exception $e) {
      $this->getLogger('rokka')->warning('Could not retrieve the file size for the thumbnail URI @uri. Error: @error',
      [
        '@uri' => $uri,
        '@error' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

}
