<?php

declare(strict_types=1);

namespace Drupal\Tests\rokka\Kernel;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\Tests\rokka\Traits\RokkaMetadataCreationTrait;
use Drupal\rokka\Entity\RokkaMetadata;
use Drupal\rokka\Entity\RokkaMetadataInterface;
use Drupal\user\UserInterface;
use PHPUnit\Framework\Attributes\CoversClass;
use PHPUnit\Framework\Attributes\Group;

/**
 * Tests the RokkaMetadata entity class.
 */
#[Group('rokka')]
#[CoversClass(RokkaMetadata::class)]
class RokkaMetadataEntityTest extends EntityKernelTestBase {

  use RokkaMetadataCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = ['rokka'];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('rokka_metadata');
  }

  /**
   * Tests getting and setting the width and height.
   */
  public function testWidthAndHeight(): void {
    // Create a new RokkaMetadata entity.
    $metadata = $this->createRokkaMetadata();

    // Test that width and height are NULL when not set.
    $this->assertNull($metadata->getWidth());
    $this->assertNull($metadata->getHeight());

    // Test setting width and height.
    $metadata->setWidth(1920);
    $metadata->setHeight(1080);

    $this->assertEquals(1920, $metadata->getWidth());
    $this->assertEquals(1080, $metadata->getHeight());

    // Save the entity and verify the values persist.
    $metadata->save();

    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);
    $this->assertEquals(1920, $reloaded_metadata->getWidth());
    $this->assertEquals(1080, $reloaded_metadata->getHeight());
  }

  /**
   * Tests that we can create an entity with custom field values.
   */
  public function testCreateWithCustomFieldValues(): void {
    $metadata = $this->createRokkaMetadata([
      'hash' => 'abcdef0123456789abcdef0123456789abcdef01',
      'binary_hash' => 'fedcba9876543210fedcba9876543210fedcba98',
      'uri' => 'rokka://custom-image.png',
      'format' => 'png',
      'filesize' => 54321,
      'width' => 800,
      'height' => 600,
      'status' => FALSE,
    ]);

    $assertions = function (RokkaMetadata $metadata) {
      $this->assertEquals('abcdef0123456789abcdef0123456789abcdef01', $metadata->getHash());
      $this->assertEquals('fedcba9876543210fedcba9876543210fedcba98', $metadata->getBinaryHash());
      $this->assertEquals('rokka://custom-image.png', $metadata->getUri());
      $this->assertEquals('png', $metadata->getFormat());
      $this->assertEquals(54321, $metadata->getFilesize());
      $this->assertEquals(800, $metadata->getWidth());
      $this->assertEquals(600, $metadata->getHeight());
      $this->assertFalse($metadata->isPublished());
    };

    $assertions($metadata);

    $metadata->save();
    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);

    $assertions($reloaded_metadata);
  }

  /**
   * Tests the methods to get and set the published status.
   */
  public function testPublishedStatus(): void {
    $metadata = $this->createRokkaMetadata();

    // By default, entities should be published.
    $this->assertTrue($metadata->isPublished());

    // Test unpublishing.
    $metadata->setUnpublished();
    $this->assertFalse($metadata->isPublished());

    // Test publishing.
    $metadata->setPublished();
    $this->assertTrue($metadata->isPublished());

    // Save and reload to verify persistence.
    $metadata->save();

    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);
    $this->assertTrue($reloaded_metadata->isPublished());

    // Verify that the unpublished state persists.
    $reloaded_metadata->setUnpublished();
    $reloaded_metadata->save();

    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);
    $this->assertFalse($reloaded_metadata->isPublished());
  }

  /**
   * Tests getting and setting the hash.
   */
  public function testHash(): void {
    $metadata = $this->createRokkaMetadata();

    // Verify the default hash from the trait.
    $this->assertEquals('0123456789abcdef0123456789abcdef01234567', $metadata->getHash());

    // Test setting a new hash.
    $new_hash = 'abcdef0123456789abcdef0123456789abcdef01';
    $metadata->setHash($new_hash);
    $this->assertEquals($new_hash, $metadata->getHash());

    // Save and reload to verify persistence.
    $metadata->save();

    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);
    $this->assertEquals($new_hash, $reloaded_metadata->getHash());
  }

  /**
   * Tests getting and setting the binary hash.
   */
  public function testBinaryHash(): void {
    $metadata = $this->createRokkaMetadata();

    // Verify the default binary hash from the trait.
    $this->assertEquals('0123456789abcdef0123456789abcdef01234567', $metadata->getBinaryHash());

    // Test setting a new binary hash.
    $new_binary_hash = 'fedcba9876543210fedcba9876543210fedcba98';
    $metadata->setBinaryHash($new_binary_hash);
    $this->assertEquals($new_binary_hash, $metadata->getBinaryHash());

    // Save and reload to verify persistence.
    $metadata->save();

    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);
    $this->assertEquals($new_binary_hash, $reloaded_metadata->getBinaryHash());
  }

  /**
   * Tests getting and setting the filesize.
   */
  public function testFilesize(): void {
    $metadata = $this->createRokkaMetadata();

    // Verify the default filesize from the trait.
    $this->assertEquals(12345, $metadata->getFilesize());

    // Test setting a new filesize.
    $new_filesize = 987654;
    $metadata->setFilesize($new_filesize);
    $this->assertEquals($new_filesize, $metadata->getFilesize());

    // Save and reload to verify persistence.
    $metadata->save();

    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);
    $this->assertEquals($new_filesize, $reloaded_metadata->getFilesize());
  }

  /**
   * Tests getting and setting the URI.
   */
  public function testUri(): void {
    $metadata = $this->createRokkaMetadata([
      'uri' => 'rokka://original-image.jpg',
    ]);

    // Verify the URI.
    $this->assertEquals('rokka://original-image.jpg', $metadata->getUri());

    // Test setting a new URI.
    $new_uri = 'rokka://updated-image.png';
    $metadata->setUri($new_uri);
    $this->assertEquals($new_uri, $metadata->getUri());

    // Save and reload to verify persistence.
    $metadata->save();

    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);
    $this->assertEquals($new_uri, $reloaded_metadata->getUri());
  }

  /**
   * Tests getting and setting the format.
   */
  public function testFormat(): void {
    $metadata = $this->createRokkaMetadata();

    // Verify the default format from the trait.
    $this->assertEquals('jpg', $metadata->getFormat());

    // Test setting a new format.
    $new_format = 'png';
    $metadata->setFormat($new_format);
    $this->assertEquals($new_format, $metadata->getFormat());

    // Save and reload to verify persistence.
    $metadata->save();

    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);
    $this->assertEquals($new_format, $reloaded_metadata->getFormat());
  }

  /**
   * Tests getting and setting the static metadata.
   */
  public function testStaticMetadata(): void {
    $metadata = $this->createRokkaMetadata();

    // Test that static metadata is empty by default.
    $this->assertEquals([], $metadata->getStaticMetadata());

    // Test setting static metadata.
    // Static metadata from Rokka is structured as an array with nested data.
    $static_metadata = [
      [
        'exif' => [
          'copyright' => 'Test Copyright',
        ],
        'auto_description' => [
          'descriptions' => [
            // cspell:disable
            'de' => 'Ein Testbild.',
            'fr' => 'Une image de test.',
            'it' => 'Un\'immagine di prova.',
            'en' => 'A test image.',
            // cspell:enable
          ],
        ],
      ],
    ];
    $metadata->setStaticMetadata($static_metadata);

    $retrieved = $metadata->getStaticMetadata();
    $this->assertEquals($static_metadata, $retrieved);

    // Save and reload to verify persistence.
    $metadata->save();

    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);
    $this->assertEquals($static_metadata, $reloaded_metadata->getStaticMetadata());
  }

  /**
   * Tests getting and setting the thumbnail flag.
   */
  public function testThumbnail(): void {
    $metadata = $this->createRokkaMetadata();

    // By default, the entity should not be a thumbnail.
    $this->assertFalse($metadata->isThumbnail());

    // Test setting the thumbnail flag to TRUE.
    $metadata->setThumbnail(TRUE);
    $this->assertTrue($metadata->isThumbnail());

    // Save and reload to verify persistence.
    $metadata->save();

    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);
    $this->assertTrue($reloaded_metadata->isThumbnail());

    // Test setting the thumbnail flag to FALSE.
    $reloaded_metadata->setThumbnail(FALSE);
    $reloaded_metadata->save();

    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);
    $this->assertFalse($reloaded_metadata->isThumbnail());
  }

  /**
   * Tests getting and setting the owner.
   */
  public function testOwner(): void {
    // Create test users.
    $user1 = $this->createUser();
    $user2 = $this->createUser();

    $metadata = $this->createRokkaMetadata();

    $assertions = function (RokkaMetadataInterface $metadata, UserInterface $expected_owner) {
      $this->assertEquals($expected_owner->id(), $metadata->getOwnerId());
      $this->assertEquals($expected_owner->id(), $metadata->getOwner()->id());
    };

    // Test setting the owner by user entity.
    $metadata->setOwner($user1);
    $assertions($metadata, $user1);

    $metadata->save();

    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);
    $assertions($reloaded_metadata, $user1);

    // Test setting the owner by user ID.
    $metadata->setOwnerId($user2->id());
    $assertions($metadata, $user2);

    $metadata->save();

    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);
    $assertions($reloaded_metadata, $user2);
  }

  /**
   * Tests getting and setting the creation time.
   */
  public function testCreatedTime(): void {
    $request_time = \Drupal::time()->getRequestTime();
    $metadata = $this->createRokkaMetadata();
    $metadata->save();

    // Verify the created time. Allow up to 10 minutes to account for slow test
    // infrastructure.
    $created_time = $metadata->getCreatedTime();
    $this->assertGreaterThanOrEqual($created_time, $request_time);
    $this->assertLessThanOrEqual($created_time + 600, $request_time);

    // Test setting a custom creation time.
    $custom_time = strtotime('2024-01-15 12:00:00');
    $metadata->setCreatedTime($custom_time);
    $this->assertEquals($custom_time, $metadata->getCreatedTime());

    // Save and reload to verify persistence.
    $metadata->save();

    $reloaded_metadata = $this->reloadEntity($metadata);
    $this->assertInstanceOf(RokkaMetadata::class, $reloaded_metadata);
    $this->assertEquals($custom_time, $reloaded_metadata->getCreatedTime());
  }

}
