<?php

namespace Drupal\role_inheritance\Service;

use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;

class RoleInheritanceMap implements RoleInheritanceMapInterface {

  public const CONFIG_NAME = 'role_inheritance.settings';

  /**
   * Role inheritance settings config.
   *
   * @var Config
   */
  protected Config $config;

  /**
   * Tree of directly inherited roles, loaded from config.
   *
   * Nested array keyed by role with sub array values for inherited roles.
   *
   * @var array|null
   */
  protected ?array $inheritanceTree = NULL;

  /**
   * Map of all inherited roles, built from config.
   *
   * Nested array keyed by role with sub array values for inherited roles.
   * 
   * This value is built on demand by `populateInheritanceMap()` when needed.
   *
   * @var array|null
   */
  protected ?array $inheritanceMap = NULL;

  /**
   * Create Inheritance map service using the role inheritance settings.
   *
   * @param ConfigFactoryInterface $config_factory
   *   Drupal Config Factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->config = $config_factory->getEditable(static::CONFIG_NAME);
  }

  /**
   * {@inheritdoc}
   */
  public function loadTree(bool $reload = FALSE): void {
    if (!$this->inheritanceTree || $reload) {
      // Load Map from config.
      $this->inheritanceTree = $this->config->get('role_map');

      if (is_null($this->inheritanceTree) || !is_array($this->inheritanceTree)) {
        // Check for invalid config.
        $this->inheritanceTree = [];
      }

      // Clean up any self references or duplicate values.
      $this->inheritanceTree = $this->sanatizeMap($this->inheritanceTree);
    }
  }

  /**
   * Clean up the map for saving.
   *
   * Prevents the following scenarios from existing in the map
   * - Self inheritance, which could create a loop.
   * - Duplicate inheritance.
   *
   * @param array $map
   *   The map to process.
   * @return array
   *   The cleaned map.
   */
  protected function sanatizeMap(array $map): array {
    foreach ($map as $parent => $children) {
      // Prevent a role from inheriting itself.
      if (in_array($parent, $children)) {
        $map[$parent] = array_diff($children, [$parent]);
      }

      // Ensure there are no duplicate values.
      $map[$parent] = array_unique($map[$parent]);
    }
    return $map;
  }

  /**
   * {@inheritdoc}
   */
  public function getTree(): array {
    if (!$this->inheritanceTree) {
      // Ensure the tree is loaded from config.
      $this->loadTree();
    }

    return $this->inheritanceTree;
  }

  /**
   * {@inheritdoc}
   */
  public function getMap(): array {
    if (!$this->inheritanceMap) {
      $this->buildMap();
    }

    return $this->inheritanceMap;
  }

  /**
   * {@inheritdoc}
   */
  public function buildMap(): void {
    if (!$this->inheritanceTree) {
      // Ensure the tree is loaded before processing.
      $this->loadTree();
    }

    $this->inheritanceMap = [];

    // If we are starting fresh on the first run, flatten each role's tree.
    foreach ($this->inheritanceTree as $role_name => $inherited_list) {
      if (!isset($this->inheritanceMap[$role_name])) {
        // Some roles may be set by recursion, we don't need to reprocess them.
        $this->populateInheritanceMap($role_name);
      }
    }
  }

  /**
   * Helper function to populate the flattened inheritance map.
   * 
   * This function will recurively flatten the inheritance tree into a map.
   * 
   * This is triggered by `buildMap()`.
   *
   * @param string $role
   *   Role to process map for.
   */
  protected function populateInheritanceMap(string $role): void {
    // If we have a role, recursively flatten inherited roles and merge data.
    if (!isset($this->inheritanceMap[$role])) {
      // Set prepare map, using the inherited role list.
      $this->inheritanceMap[$role] = $this->inheritanceTree[$role] ?? [];
    }

    // Process Inherited roles, and merge mappings.
    $inherited_roles = $this->inheritanceTree[$role] ?? [];
    foreach ($inherited_roles as $inherited_role) {
      if (!isset($this->inheritanceMap[$inherited_role])) {
        // If Inherited role hasn't been processed, process it.
        $this->populateInheritanceMap($inherited_role);
      }

      // Merge inherited childern into parent.
      $this->inheritanceMap[$role] = array_unique(array_merge($this->inheritanceMap[$role], $this->inheritanceMap[$inherited_role]));
    }

    if (in_array($role, $this->inheritanceMap[$role])) {
      $this->inheritanceMap[$role] = array_diff($this->inheritanceMap[$role], [$role]);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getDirectlyInheritedRoles(string $role): array {
    if (!$this->inheritanceTree) {
      $this->loadTree();
    }
    return $this->inheritanceTree[$role] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function getAllInheritedRoles(string $role): array {
    if (!$this->inheritanceMap) {
      $this->buildMap();
    }
    return $this->inheritanceMap[$role] ?? [];
  }

  /**
   * {@inheritdoc}
   */
  public function extendRoles(array $initial_roles, bool $include_initial = TRUE): array {
    $inherited = [];
    foreach ($initial_roles as $role_name) {
      $additional_roles = $this->getAllInheritedRoles($role_name);
      $inherited = array_unique(array_merge($inherited, $additional_roles));
    }

    if ($include_initial) {
      return array_unique(array_merge($inherited, $initial_roles));
    }
    else {
      return array_diff($inherited, $initial_roles);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function pruneInheritedRoles(array $roles): array {
    if (!$this->inheritanceMap) {
      $this->buildMap();
    }

    $all_children = $this->extendRoles($roles, FALSE);
    return array_diff($roles, $all_children);
  }

  /**
   * {@inheritdoc}
   */
  public function pruneParentRoles(array $roles): array {
    if (!$this->inheritanceMap) {
      $this->buildMap();
    }

    $all_parents = $this->getExtendedParents($roles, FALSE);
    return array_diff($roles, $all_parents);
  }

  /**
   * {@inheritdoc}
   */
  public function getDirectParents(string $role): array {
    if (!$this->inheritanceTree) {
      $this->loadTree();
    }

    $parents = [];
    foreach ($this->inheritanceTree as $parent_name => $inherited_roles) {
      if (in_array($role, $inherited_roles)) {
        $parents[] = $parent_name;
      }
    }
    return $parents;
  }

  /**
   * {@inheritdoc}
   */
  public function getAllParents(string $role): array {
    if (!$this->inheritanceMap) {
      $this->buildMap();
    }

    $parents = [];
    foreach ($this->inheritanceMap as $parent_name => $inherited_roles) {
      if (in_array($role, $inherited_roles)) {
        $parents[] = $parent_name;
      }
    }
    return $parents;
  }

  /**
   * {@inheritdoc}
   */
  public function getExtendedParents(array $roles, bool $include_initial = TRUE): array {
    if (!$this->inheritanceMap) {
      $this->buildMap();
    }

    $parents = [];
    foreach ($this->inheritanceMap as $parent_name => $inherited_roles) {
      if (array_intersect($roles, $inherited_roles)) {
        $parents[] = $parent_name;
      }
    }

    if ($include_initial) {
      return array_unique(array_merge($parents, $roles));
    }
    else {
      return array_diff($parents, $roles);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function updateMap(array $tree): void {
    // Remove loops and duplicate value from new tree.
    $tree = $this->sanatizeMap($tree);

    // Remove empty maps, we don't need to save empty arrays to config.
    $tree = array_filter($tree);

    // Update and save config.
    $this->config->set('role_map', $tree);
    $this->config->save();

    // Reset the tree and map with new value.
    $this->inheritanceTree = $tree;
    $this->inheritanceMap = NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function updateInheritedRoles(string $role_name, array $inherited_roles): void {
    $tree = $this->getTree();
    $tree[$role_name] = $inherited_roles;
    $this->updateMap($tree);
  }

}
