<?php

namespace Drupal\role_inheritance\Service;

interface RoleInheritanceMapInterface {

  /**
   * Load the tree from config.
   *
   * @return void
   */
  public function loadTree(): void;

  /**
   * Fetch the tree.
   *
   * If tree is not loaded from config, it will be loaded via `loadTree()`.
   *
   * @return array
   *   A list of role which directly inherit from each role.
   */
  public function getTree(): array;

  /**
   * Fetch the role inheritance mapping.
   *
   * If the role inheritance mapping isn't built yet, it will be. We also
   * ensure that the full tree is already loading from config via `loadTree()`.
   *
   * @return array
   *   A list of all role which directly and indirectly inherit from each role.
   */
  public function getMap(): array;

  /**
   * Build the inheritance map.
   *
   * This will process the tree and flatten the values into a map. Map can be
   * retrieved with `getMap()`.
   */
  public function buildMap(): void;

  /**
   * Get a list of role ids which inhert directly from a role.
   *
   * @param string $role
   *   The role to return records for.
   *
   * @return array
   *   Role ids directly inherited from the specified role.
   */
  public function getDirectlyInheritedRoles(string $role): array;

  /**
   * Get a list of role ids which inhert directly from a role.
   *
   * @param string $role
   *   The role to return records for.
   *
   * @return array
   *   All role ids directly and indirectly inherited from the specified role.
   */
  public function getAllInheritedRoles(string $role): array;

  /**
   * Find inherited roles for multiple roles.
   *
   * @param array $initial_roles
   *   The list of roles to expaned.
   * @param bool $inclusive
   *   TRUE to include the initial_roles in the return. False to exclude them.
   *
   * @return array
   *   The list of expanded roles.
   */
  public function extendRoles(array $initial_roles, bool $include_initial = TRUE): array;

  /**
   * Simplify a list of roles, to remove roles that are inherited from others.
   *
   * Removes roles that are inherited from one or more roles in the original role
   * list, to provide the smallest list of roles needed to achieve the same list
   * when inheritance is applied.
   *
   * Usage Example: Users only need the highest roles in a tree and via role
   * inheritance roles below those will also be given to the user.
   *
   * @param array $roles
   *   Role list to process.
   * @return array
   *   Simplifed role list.
   */
  public function pruneInheritedRoles(array $roles): array;

  /**
   * Simplify a list of roles, to remove roles that receive inherited info.
   *
   * Removes roles from the list if one or more roles inherit form the role.
   *
   * Usage Example: Content only needs to have lowest roles in the tree and
   * through role inheritance, the other roles higher will be provided.
   *
   * @param array $roles
   *   Role list to process.
   * @return array
   *   Simplifed role list.
   */
  public function pruneParentRoles(array $roles): array;

  /**
   * Find roles which directly inherit a role.
   *
   * @param string $role
   *   The role to search for.
   *
   * @return array
   *   The list of parent roles.
   */
  public function getDirectParents(string $role): array;

  /**
   * Find roles which directly or indirectly inherit a role.
   *
   * @param string $role
   *   The role to search for.
   *
   * @return array
   *   The list of parent roles.
   */
  public function getAllParents(string $role): array;

  /**
   * Find roles which directly or indirectly inherit any supplied roles.
   *
   * @param array $roles
   *   The roles to search for.
   *
   * @return array
   *   The list of parent roles.
   */
  public function getExtendedParents(array $roles): array;

  /**
   * Update and save the map to config.
   *
   * @param array $map
   *   The new role inheritance map.
   */
  public function updateMap(array $map): void;

  /**
   * Updates the inherited roles for a role.
   *
   * This method will update the inheritance map and save the config.
   *
   * @param string $role_name
   *   Role to update in map.
   * @param array $inherited_roles
   *   Update role list.
   */
  public function updateInheritedRoles(string $role_name, array $inherited_roles): void;

}
