<?php

namespace Drupal\role_request\Entity;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityAccessControlHandler;
use Drupal\Core\Entity\EntityHandlerInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Role request access handler.
 */
final class AccessHandler extends EntityAccessControlHandler implements EntityHandlerInterface {

  /**
   * Constructs an access control handler instance.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config
   *   The config factory.
   */
  public function __construct(
    EntityTypeInterface $entity_type,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected ConfigFactoryInterface $config,
  ) {
    parent::__construct($entity_type);
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type): self {
    return new static(
      $entity_type,
      $container->get('entity_type.manager'),
      $container->get('config.factory')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function checkCreateAccess(AccountInterface $account, array $context, $entity_bundle = NULL): AccessResultInterface {
    if (
      // Safeguard to avoid anonymous users creating role requests.
      $account->isAuthenticated() &&
      $account->hasPermission('request role') &&
      $this->hasOpenRequestsBelowLimit($account) &&
      $this->hasClosedRequestsBelowLimit($account)
    ) {
      return AccessResult::allowed()
        ->cachePerPermissions()
        ->cachePerUser()
        ->addCacheableDependency($this->config->get('role_request.settings'))
        ->addCacheTags(['role_request_list']);
    }
    else {
      // No opinion.
      return AccessResult::neutral()
        ->addCacheableDependency($this->config->get('role_request.settings'))
        ->addCacheTags(['role_request_list']);
    }
  }

  /**
   * Request pending review are lower than the config limit.
   */
  protected function hasOpenRequestsBelowLimit(AccountInterface $account): bool {
    $query = $this->entityTypeManager->getStorage('role_request')->getQuery();
    $count = $query
      ->accessCheck(FALSE)
      ->condition('uid', $account->id())
      ->notExists('status')
      ->count()
      ->execute();
    return $count < $this->config->get('role_request.settings')->get('open_requests_per_user');
  }

  /**
   * Reviewed requests are lower than the config limit.
   */
  protected function hasClosedRequestsBelowLimit(AccountInterface $account): bool {
    $query = $this->entityTypeManager->getStorage('role_request')->getQuery();
    $count = $query
      ->accessCheck(FALSE)
      ->condition('uid', $account->id())
      ->exists('status')
      ->count()
      ->execute();
    return $count < $this->config->get('role_request.settings')->get('closed_requests_per_user');
  }

}
