<?php

namespace Drupal\role_request\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\RoleInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Config form for general settings.
 */
final class Settings extends ConfigFormBase implements ContainerInjectionInterface {

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'role_request_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['role_request.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typedConfigManager,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('role_request.settings');

    $form['options'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Role request options'),
    ];

    $form['options']['roles_per_request'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum number of roles per request'),
      '#default_value' => $config->get('roles_per_request'),
      '#description' => $this->t('Choosing "1" will allow for only one role per request'),
      '#required' => TRUE,
    ];

    $form['options']['open_requests_per_user'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum "pending review" requests per user'),
      '#default_value' => $config->get('open_requests_per_user'),
      '#description' => $this->t('How many requests each user can submit before they get approved or denied.'),
      '#required' => TRUE,
    ];

    $form['options']['closed_requests_per_user'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum approved or denied requests per user'),
      '#default_value' => $config->get('closed_requests_per_user'),
      '#description' => $this->t('Choosing "1" means that a user that gets a request denied can not request again unless the request is deleted.'),
      '#required' => TRUE,
    ];

    $form['options']['allow_user_message'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Allow request message'),
      '#description' => $this->t('Allows requesters to submit a message along with each request, explaining why they need the role.'),
      '#default_value' => $config->get('allow_user_message'),
    ];

    // Get all user roles, excluding anonymous and authenticated.
    $roleEntities = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    unset($roleEntities[RoleInterface::ANONYMOUS_ID]);
    unset($roleEntities[RoleInterface::AUTHENTICATED_ID]);
    $roles = array_map(fn(RoleInterface $role) => $role->label(), $roleEntities);

    $form['options']['allowed_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Roles'),
      '#options' => $roles,
      '#default_value' => $config->get('allowed_roles'),
      '#description' => $this->t('Select the roles that users will be able to request.'),
      '#required' => TRUE,
    ];

    $form['options']['role_request_description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Role request description/instructions'),
      '#description' => $this->t('Text entered here will be displayed in the request role form'),
      '#default_value' => $config->get('role_request_description'),
    ];

    $form['email'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Role request email options'),
      '#description' => $this->t('Configure emails that role request can send, including enabled/disabled status.'),
    ];

    $form['email']['role_request_email_admin_content'] = [
      '#type' => 'details',
      '#title' => $this->t('Admin email settings'),
      '#open' => FALSE,
    ];

    $form['email']['role_request_email_admin_content']['send_email_to_admin'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Send administrators email on new requests'),
      '#default_value' => $config->get('send_email_to_admin'),
    ];

    $form['email']['role_request_email_admin_content']['admin_email_addresses'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Admin email addressess'),
      '#description' => $this->t('A comma-separated list of emails -OR- Leave blank to use site admin (UID 1) email address.'),
      '#default_value' => $config->get('admin_email_addresses'),
      '#size' => 60,
      '#maxlength' => 128,
    ];
    $form['email']['role_request_email_admin_content']['admin_email_subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Admin email subject'),
      '#default_value' => $config->get('admin_email_subject'),
      '#size' => 60,
      '#maxlength' => 128,
    ];
    $form['email']['role_request_email_admin_content']['admin_email_body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Admin email message body'),
      '#description' => $this->t('Tokens like [site:base-url] can be used. Use [user:account-name] for user name, and [role-request:roles] for role requests.'),
      '#default_value' => $config->get('admin_email_body'),
      '#size' => 60,
    ];

    $form['email']['send_approve_email_content'] = [
      '#type' => 'details',
      '#title' => $this->t('User Approval Email Settings'),
      '#open' => FALSE,
    ];
    $form['email']['send_approve_email_content']['send_user_approval_email'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Send email on approval'),
      '#default_value' => $config->get('send_user_approval_email'),
    ];

    $form['email']['send_approve_email_content']['send_user_approval_subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#default_value' => $config->get('send_user_approval_subject'),
      '#size' => 60,
      '#maxlength' => 128,
    ];
    $form['email']['send_approve_email_content']['send_user_approval_body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Message Body'),
      '#description' => $this->t('Tokens like [site:base-url] can be used. Use [role-request:roles] for approved role(s).'),
      '#default_value' => $config->get('send_user_approval_body'),
      '#size' => 60,
    ];

    $form['email']['send_denial_email_content'] = [
      '#type' => 'details',
      '#title' => $this->t('Role Request Denied Email Settings'),
      '#open' => FALSE,
    ];

    $form['email']['send_denial_email_content']['send_user_deny_email'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Send email on denial'),
      '#default_value' => $config->get('send_user_deny_email'),
    ];

    $form['email']['send_denial_email_content']['send_user_deny_subject'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subject'),
      '#default_value' => $config->get('send_user_deny_subject'),
      '#size' => 60,
      '#maxlength' => 128,
    ];
    $form['email']['send_denial_email_content']['send_user_deny_body'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Message Body'),
      '#description' => $this->t('Body of the email. Tokens like [site:base-url] can be used. Use [role-request:roles] for denied role(s).'),
      '#default_value' => $config->get('send_user_deny_body'),
      '#size' => 60,
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    if (intval($form_state->getValue('roles_per_request')) < 1) {
      $form_state->setErrorByName('roles_per_request', $this->t('At least one role needs to be allowed'));
    }
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('role_request.settings');

    $submitted_values = $form_state->cleanValues()->getValues();

    foreach ($submitted_values as $submitted_value_key => $submitted_value) {
      $config->set($submitted_value_key, $submitted_value);
    }

    $config->save();

    parent::submitForm($form, $form_state);
  }

}
