<?php

namespace Drupal\role_request\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Mail\MailManagerInterface;
use Drupal\Core\Utility\Token;
use Drupal\Component\Render\PlainTextOutput;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\role_request\RoleRequestInterface;

/**
 * Hook implementations for mail.
 */
class Mail {

  public function __construct(
    protected Token $token,
    protected LanguageManagerInterface $languageManager,
    protected ConfigFactoryInterface $config,
    protected MailManagerInterface $mailManager,
  ) {
  }

  /**
   * Implements hook_mail().
   */
  #[Hook('mail')]
  public function mail($key, &$message, $params): void {
    $langcode = $message['langcode'];
    $language = $this->languageManager->getLanguage($langcode);
    $original_language = $this->languageManager->getConfigOverrideLanguage();
    $this->languageManager->setConfigOverrideLanguage($language);
    $settings = $this->config->get('role_request.settings');
    $tokenOptions = ['langcode' => $langcode, 'clear' => TRUE];

    $message['from'] = $this->config->get('system.site')->get('mail');
    $message['subject'] .= PlainTextOutput::renderFromHtml($this->token->replace($settings->get($key . '_subject'), $params, $tokenOptions));
    $message['body'][] = $this->token->replace($settings->get($key . '_body'), $params, $tokenOptions);
    $this->languageManager->setConfigOverrideLanguage($original_language);
  }

  /**
   * Implements hook_ENTITY_TYPE_insert().
   */
  #[Hook('role_request_insert')]
  public function adminMail(RoleRequestInterface $entity): void {
    $settings = $this->config->get('role_request.settings');
    if (!$settings->get('send_email_to_admin')) {
      return;
    }
    $module = 'role_request';
    $key = 'admin_email';
    $to = $settings->get('admin_email_addresses');
    $params = [
      'user' => $entity->getOwner(),
      'role-request' => $entity,
    ];

    // There is no config preference in the module to chose the language
    // of the admin email so it is sent in the language of the requester.
    $langcode = $entity->getOwner()->getPreferredLangcode();
    $send = TRUE;

    $this->mailManager->mail($module, $key, $to, $langcode, $params, NULL, $send);
  }

  /**
   * Implements hook_ENTITY_TYPE_update().
   */
  #[Hook('role_request_update')]
  public function userNotifyMail(RoleRequestInterface $entity): void {
    $settings = $this->config->get('role_request.settings');
    $send = FALSE;
    $key = '';
    if ($settings->get('send_user_approval_email') && $entity->isBeingApproved()) {
      $key = 'send_user_approval';
    }
    if ($settings->get('send_user_deny_email') && $entity->isBeingDenied()) {
      $key = 'send_user_deny';
    }
    if (!$key) {
      return;
    }
    $send = TRUE;
    $module = 'role_request';
    $to = $entity->getOwner()->getEmail();
    $langcode = $entity->getOwner()->getPreferredLangcode();
    $params = [
      'user' => $entity->getOwner(),
      'role-request' => $entity,
    ];

    $this->mailManager->mail($module, $key, $to, $langcode, $params, NULL, $send);
  }

}
