<?php

declare(strict_types=1);

namespace Drupal\role_request;

use Drupal\Component\Serialization\Json;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Entity\Query\QueryInterface;

/**
 * Provides a list controller for the role request entity type.
 */
final class RoleRequestListBuilder extends EntityListBuilder {

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    $header = [];
    $header['uid'] = $this->t('Requester');
    $header['roles'] = $this->t('Roles');
    $header['status'] = $this->t('Status');
    $header['created'] = $this->t('Created');
    $header['changed'] = $this->t('Updated');
    $header['id'] = $this->t('Request ID');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array {
    $row = [];
    $username_options = [
      'label' => 'hidden',
      'settings' => ['link' => TRUE],
    ];
    /** @var \Drupal\role_request\RoleRequestInterface $entity */
    $row['uid']['data'] = $entity->get('uid')->view($username_options);

    $row['roles']['data'] = $entity->get('roles')->view(['label' => 'hidden']);

    $statusLabel = $this->t('Pending review');
    $status = $entity->get('status')->value;
    if ($status === '1') {
      $statusLabel = $this->t('Approved');
    }
    if ($status === '0') {
      $statusLabel = $this->t('Denied');
    }
    $row['status'] = $statusLabel;

    $row['created']['data'] = $entity->get('created')->view(['label' => 'hidden']);
    $row['changed']['data'] = $entity->get('changed')->view(['label' => 'hidden']);
    $row['id'] = $entity->id();
    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  protected function getDefaultOperations(EntityInterface $entity): array {
    $operations = parent::getDefaultOperations($entity);
    if (!$entity instanceof RoleRequestInterface) {
      return $operations;
    }

    if ($entity->access('update') && $entity->hasLinkTemplate('approve-form') && $entity->get('status')->value !== '1') {
      $url = $this->ensureDestination($entity->toUrl('approve-form'));
      $label = $this->t('Approve role request @entity_id', [
        '@entity_id' => $entity->id(),
      ]);
      $attributes = $url->getOption('attributes') ?: [];
      $attributes += [
        'aria-label' => $label,
      ];
      $url->setOption('attributes', $attributes);
      $operations['approve'] = [
        'title' => $this->t('Approve'),
        'weight' => 3,
        'url' => $url,
        'attributes' => [
          'class' => [
            'use-ajax',
          ],
          'data-dialog-type' => 'modal',
          'data-dialog-options' => Json::encode([
            'width' => 880,
          ]),
        ],
      ];
    }

    if ($entity->access('update') && $entity->hasLinkTemplate('deny-form') && $entity->get('status')->value !== '0') {
      $url = $this->ensureDestination($entity->toUrl('deny-form'));
      $label = $this->t('Deny role request @entity_id', [
        '@entity_id' => $entity->id(),
      ]);
      $attributes = $url->getOption('attributes') ?: [];
      $attributes += [
        'aria-label' => $label,
      ];
      $url->setOption('attributes', $attributes);
      $operations['deny'] = [
        'title' => $this->t('Deny'),
        'weight' => 5,
        'url' => $url,
        'attributes' => [
          'class' => [
            'use-ajax',
          ],
          'data-dialog-type' => 'modal',
          'data-dialog-options' => Json::encode([
            'width' => 880,
          ]),
        ],
      ];
    }
    return $operations;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEntityListQuery(): QueryInterface {
    return parent::getEntityListQuery()->sort('id', 'DESC');
  }

}
