<?php

namespace Drupal\role_request\Entity;

use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\ContentEntityDeleteForm;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\Form\DeleteMultipleForm;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\role_request\Form\RoleRequestApproveForm;
use Drupal\role_request\Form\RoleRequestDenyForm;
use Drupal\role_request\Form\RoleRequestEditForm;
use Drupal\role_request\Form\RoleRequestForm;
use Drupal\role_request\RoleRequestInterface;
use Drupal\role_request\RoleRequestListBuilder;
use Drupal\user\EntityOwnerTrait;
use Drupal\views\EntityViewsData;

/**
 * Defines the role request entity class.
 */
#[ContentEntityType(
  id: "role_request",
  label: new TranslatableMarkup("Role request"),
  label_collection: new TranslatableMarkup("Role requests"),
  label_singular: new TranslatableMarkup("role request"),
  label_plural: new TranslatableMarkup("role requests"),
  label_count: [
    'singular' => '@count role request',
    'plural' => '@count role requests',
  ],
  entity_keys: [
    'id' => 'id',
    'uuid' => 'uuid',
    'label' => 'id',
    'owner' => 'uid',
  ],
  handlers: [
    'list_builder' => RoleRequestListBuilder::class,
    'views_data' => EntityViewsData::class,
    'access' => AccessHandler::class,
    'form' => [
      'add' => RoleRequestForm::class,
      'edit' => RoleRequestEditForm::class,
      'delete' => ContentEntityDeleteForm::class,
      'delete-multiple-confirm' => DeleteMultipleForm::class,
      'approve' => RoleRequestApproveForm::class,
      'deny' => RoleRequestDenyForm::class,
    ],
    'route_provider' => [
      'html' => RouteProvider::class,
    ],
  ],
  links: [
    'collection' => '/admin/people/role-request',
    'add-form' => '/user/role-request-add',
    'canonical' => '/admin/people/role-request/{role_request}',
    'delete-form' => '/admin/people/role-request/{role_request}/delete',
    'edit-form' => '/admin/people/role-request/{role_request}/edit',
    'delete-multiple-form' => '/admin/content/role-request/delete-multiple',
    'approve-form' => '/admin/people/role-request/{role_request}/approve',
    'deny-form' => '/admin/people/role-request/{role_request}/deny',
  ],
  admin_permission: 'administer role requests',
  base_table: 'role_request',
)]
final class RoleRequest extends ContentEntityBase implements RoleRequestInterface {

  use EntityChangedTrait;
  use EntityOwnerTrait;

  /**
   * {@inheritdoc}
   */
  public function preSave(EntityStorageInterface $storage): void {
    parent::preSave($storage);
    if (!$this->getOwnerId()) {
      // If no owner has been set explicitly, make the current user the owner.
      $this->setOwnerId(\Drupal::currentUser()->id());
    }
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityStorageInterface $storage, $update = TRUE): void {
    parent::postSave($storage);

    // When the request is approved and it was not already approved,
    // grant the roles to the user.
    if ($this->get('status')->value === '1' && $this->original->get('status')->value !== $this->get('status')->value) {
      $user = $this->getOwner();
      foreach ($this->get('roles')->getValue() as $role) {
        $user->addRole($role['target_id']);
      }
      $user->save();
    }
  }

  /**
   * {@inheritdoc}
   */
  public function isBeingApproved(): bool {
    return $this->get('status')->value === '1' && $this->original->get('status')->value !== $this->get('status')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function isBeingDenied(): bool {
    return $this->get('status')->value === '0' && $this->original->get('status')->value !== $this->get('status')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime(): int {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Status'))
      ->setSetting('on_label', 'Approved')
      ->setSetting('off_label', 'Denied')
      ->setDisplayOptions('form', [
        'type' => 'options_buttons',
        'settings' => [],
        'weight' => 15,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'boolean',
        'label' => 'above',
        'weight' => 15,
        'settings' => [
          'format' => 'enabled-disabled',
        ],
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['message'] = BaseFieldDefinition::create('text_long')
      ->setLabel(t('Request message'))
      ->setDisplayOptions('form', [
        'type' => 'text_textarea',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'text_default',
        'label' => 'above',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    // When the request is being edited a message for the user can be added
    // to include in the email, for example.
    $fields['message_for_user'] = BaseFieldDefinition::create('text_long')
      ->setLabel(t('Message for user'))
      ->setDisplayOptions('form', [
        'type' => 'text_textarea',
        'weight' => 25,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'text_default',
        'label' => 'above',
        'weight' => 25,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Requester'))
      ->setDescription(t('The user that submitted the request.'))
      ->setSetting('target_type', 'user')
      ->setDefaultValueCallback(self::class . '::getDefaultEntityOwner')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 15,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['roles'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Choose roles to request'))
      ->setCardinality(BaseFieldDefinition::CARDINALITY_UNLIMITED)
      ->setSetting('target_type', 'user_role')
      ->setRequired(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'options_buttons',
        'settings' => [],
        'weight' => 10,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'author',
        'weight' => 15,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Authored on'))
      ->setDescription(t('The time that the role request was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the role request was last edited.'));

    return $fields;
  }

}
