<?php

namespace Drupal\role_request\Form;

use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\RoleInterface;

/**
 * Form controller for the role request entity edit forms.
 */
final class RoleRequestEditForm extends ContentEntityForm {

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    unset($form['roles']['widget']['#options'][RoleInterface::ANONYMOUS_ID]);
    unset($form['roles']['widget']['#options'][RoleInterface::AUTHENTICATED_ID]);

    // Replace N/A for a more appropiate text.
    $form['status']['widget']['#options']['_none'] = $this->t('Pending review');

    // Mark the NULL status in the widget.
    if ($form['status']['widget']['#default_value'] === NULL) {
      $form['status']['widget']['#default_value'] = '_none';
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $result = parent::save($form, $form_state);

    $message_args = ['%label' => $this->entity->toLink()->toString()];
    $logger_args = [
      '%label' => $this->entity->label(),
      'link' => $this->entity->toLink($this->t('View'))->toString(),
    ];

    switch ($result) {
      case SAVED_NEW:
        $this->messenger()->addStatus($this->t('New role request %label has been created.', $message_args));
        $this->logger('role_request')->notice('New role request %label has been created.', $logger_args);
        break;

      case SAVED_UPDATED:
        $this->messenger()->addStatus($this->t('The role request %label has been updated.', $message_args));
        $this->logger('role_request')->notice('The role request %label has been updated.', $logger_args);
        break;

      default:
        throw new \LogicException('Could not save the entity.');
    }

    $form_state->setRedirectUrl($this->entity->toUrl());

    return $result;
  }

}
