<?php

namespace Drupal\role_request\Form;

use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\Display\EntityFormDisplayInterface;
use Drupal\Core\Entity\Entity\EntityFormDisplay;
use Drupal\Core\Form\FormStateInterface;
use Drupal\user\RoleInterface;

/**
 * Form controller for the role request.
 */
final class RoleRequestForm extends ContentEntityForm {

  /**
   * {@inheritdoc}
   */
  public function getFormDisplay(FormStateInterface $form_state): EntityFormDisplayInterface {
    // Create the form display on the fly.
    $config = [
      'content' => [
        'roles' => [
          'type' => 'options_buttons',
          'settings' => [],
          'weight' => 0,
          'third_party_settings' => [],
        ],
      ],
      'targetEntityType' => 'role_request',
      'bundle' => 'role_request',
    ];
    if ($this->config('role_request.settings')->get('allow_user_message')) {
      $config['content']['message'] = [
        'type' => 'text_textarea',
        'settings' => [],
        'weight' => 0,
        'third_party_settings' => [],
      ];
    }

    return EntityFormDisplay::create($config);
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    $settings = $this->config('role_request.settings');

    $form['description'] = [
      '#markup' => $settings->get('role_request_description'),
      '#weight' => -10,
    ];

    // Use radios if only one role can be requested.
    if ($settings->get('roles_per_request') <= 1) {
      $form['roles']['widget']['#type'] = 'radios';
    }
    unset($form['roles']['widget']['#options'][RoleInterface::ANONYMOUS_ID]);
    unset($form['roles']['widget']['#options'][RoleInterface::AUTHENTICATED_ID]);

    $allowedRoles = $settings->get('allowed_roles');
    foreach (array_keys($form['roles']['widget']['#options']) as $roleId) {
      if (in_array($roleId, $allowedRoles)) {
        continue;
      }
      unset($form['roles']['widget']['#options'][$roleId]);
    }

    $rolesAlreadyGranted = array_intersect(
      array_keys($form['roles']['widget']['#options']),
      $this->currentUser()->getRoles()
    );
    if ($rolesAlreadyGranted) {
      $form['roles_granted'] = [
        '#theme' => 'item_list',
        '#list_type' => 'ul',
        '#title' => 'Roles not available because they are already granted',
        '#items' => [],
        '#attributes' => ['class' => 'role-request-granted-roles'],
        '#wrapper_attributes' => ['class' => 'container'],
      ];
      foreach ($rolesAlreadyGranted as $roleId) {
        $form['roles_granted']['#items'][] = $form['roles']['widget']['#options'][$roleId];
        unset($form['roles']['widget']['#options'][$roleId]);
      }

    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state): array {
    $actions = parent::actions($form, $form_state);
    $actions['submit']['#value'] = $this->t('Request role(s)');
    return $actions;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): ContentEntityInterface {
    $entity = parent::validateForm($form, $form_state);
    $max = $this->config('role_request.settings')->get('roles_per_request');
    if ($max > 1 && count($form_state->getValue('roles')) > $max) {
      $form_state->setErrorByName('roles', $this->t('You can only request @max roles', ['@max' => $max]));
    }
    return $entity;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $result = parent::save($form, $form_state);

    /** @var \Drupal\role_request\Entity\RoleRequest */
    $roleRequest = $this->entity;

    $logger_args = [
      '%label' => $this->entity->label(),
      'link' => $this->entity->toLink($this->t('View'))->toString(),
    ];

    switch ($result) {
      case SAVED_NEW:
        $this->messenger()->addStatus($this->t('Role requested.'));
        $this->logger('role_request')->notice('New role request %label has been created.', $logger_args);
        break;

      default:
        throw new \LogicException('Could not request the role.');
    }

    $form_state->setRedirectUrl($roleRequest->getOwner()->toUrl());

    return $result;
  }

}
