<?php

namespace Drupal\role_request\Hook;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Utility\Token;
use Drupal\role_request\RoleRequestInterface;
use Drupal\Core\Datetime\Entity\DateFormat;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\Hook\Attribute\Hook;

/**
 * Hook implementations for user.
 */
class Tokens {

  use StringTranslationTrait;

  public function __construct(
    protected Token $token,
    protected LanguageManagerInterface $languageManager,
    protected DateFormatterInterface $dateFormatter,
  ) {
  }

  /**
   * Implements hook_token_info().
   */
  #[Hook('token_info')]
  public function tokenInfo(): array {
    $types = [];
    $types['role-request'] = [
      'name' => $this->t('Role requests'),
      'description' => $this->t('Tokens related to role requests.'),
      'needs-data' => 'role-request',
    ];

    $roleRequest = [];
    $roleRequest['id'] = [
      'name' => $this->t('Role request ID'),
      'description' => $this->t('The unique ID of the role request.'),
    ];
    $roleRequest['edit-url'] = [
      'name' => $this->t('Edit URL'),
      'description' => $this->t('The URL of the request edit page.'),
    ];
    $roleRequest['roles'] = [
      'name' => $this->t('Roles requested'),
      'description' => $this->t('A comma separated list of the roles being requested.'),
    ];
    $roleRequest['message'] = [
      'name' => $this->t('User message'),
      'description' => $this->t('The text writen by the user to explain the request.'),
    ];
    $roleRequest['message-for-user'] = [
      'name' => $this->t('Message for user'),
      'description' => $this->t('The text writen by the admin to explain the approval or deny.'),
    ];
    $roleRequest['created'] = [
      'name' => $this->t('Created'),
      'description' => $this->t('The date the role request was created.'),
      'type' => 'date',
    ];
    return ['types' => $types, 'tokens' => ['user' => $roleRequest]];
  }

  /**
   * Implements hook_tokens().
   */
  #[Hook('tokens')]
  public function tokens($type, $tokens, array $data, array $options, BubbleableMetadata $bubbleable_metadata): array {
    $url_options = ['absolute' => TRUE];
    if (isset($options['langcode'])) {
      $url_options['language'] = $this->languageManager->getLanguage($options['langcode']);
      $langcode = $options['langcode'];
    }
    else {
      $langcode = NULL;
    }
    $replacements = [];
    if ($type !== 'role-request' || empty($data['role-request']) || !$data['role-request'] instanceof RoleRequestInterface) {
      return $replacements;
    }
    $roleRequest = $data['role-request'];
    foreach ($tokens as $name => $original) {
      switch ($name) {
        case 'id':
          // In the case of presave hooks id is not set yet.
          $replacements[$original] = $roleRequest->id() ?: $this->t('not yet assigned');
          break;

        case 'edit-url':
          $replacements[$original] = $roleRequest->id() ? $roleRequest->toUrl('edit-form', $url_options)->toString() : $this->t('not yet assigned');
          break;

        case 'roles':
          $roles = $roleRequest->get('roles')->referencedEntities();
          $replacements[$original] = $roles ?
            implode(', ', array_map(function ($role) {
              return $role->label();
            }, $roles)) :
            $this->t('No roles');
          break;

        case 'message':
          $replacements[$original] = $roleRequest->get('message')->value;
          break;

        case 'message-for-user':
          $replacements[$original] = $roleRequest->get('message_for_user')->value;
          break;

        case 'created':
          $date_format = DateFormat::load('medium');
          $bubbleable_metadata->addCacheableDependency($date_format);
          $replacements[$original] = $roleRequest->getCreatedTime() ? $this->dateFormatter->format($roleRequest->getCreatedTime(), 'medium', '', NULL, $langcode) : $this->t('not yet created');
          break;
      }
    }
    if ($createdTokens = $this->token->findWithPrefix($tokens, 'created')) {
      $replacements += $this->token->generate('date', $createdTokens, ['date' => $roleRequest->getCreatedTime()], $options, $bubbleable_metadata);
    }
    return $replacements;
  }

}
