<?php

namespace Drupal\role_watchdog\Entity;

use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\user\UserInterface;

/**
 * Defines the Role Watchdog entity.
 *
 * @ingroup role_watchdog
 *
 * @ContentEntityType(
 *   id = "role_watchdog",
 *   label = @Translation("Role Watchdog"),
 *   handlers = {
 *     "view_builder" = "Drupal\Core\Entity\EntityViewBuilder",
 *     "list_builder" = "Drupal\role_watchdog\RoleWatchdogListBuilder",
 *     "views_data" = "Drupal\role_watchdog\Entity\RoleWatchdogViewsData",
 *     "translation" = "Drupal\role_watchdog\RoleWatchdogTranslationHandler",
 *
 *     "form" = {
 *       "default" = "Drupal\role_watchdog\Form\RoleWatchdogForm",
 *       "add" = "Drupal\role_watchdog\Form\RoleWatchdogForm",
 *       "edit" = "Drupal\role_watchdog\Form\RoleWatchdogForm",
 *       "delete" = "Drupal\role_watchdog\Form\RoleWatchdogDeleteForm",
 *     },
 *     "access" = "Drupal\role_watchdog\RoleWatchdogAccessControlHandler",
 *     "route_provider" = {
 *       "html" = "Drupal\role_watchdog\RoleWatchdogHtmlRouteProvider",
 *     },
 *   },
 *   base_table = "role_watchdog",
 *   data_table = "role_watchdog_field_data",
 *   translatable = TRUE,
 *   admin_permission = "administer role watchdog entities",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "name",
 *     "uuid" = "uuid",
 *     "uid" = "user_id",
 *     "langcode" = "langcode",
 *     "status" = "status",
 *   },
 *   links = {
 *     "canonical" = "/admin/structure/role_watchdog/{role_watchdog}",
 *     "add-form" = "/admin/structure/role_watchdog/add",
 *     "edit-form" = "/admin/structure/role_watchdog/{role_watchdog}/edit",
 *     "delete-form" = "/admin/structure/role_watchdog/{role_watchdog}/delete",
 *     "collection" = "/admin/structure/role_watchdog",
 *   },
 *   field_ui_base_route = "role_watchdog.settings"
 * )
 */
class RoleWatchdog extends ContentEntityBase implements RoleWatchdogInterface {

  use EntityChangedTrait;

  const ROLE_REMOVED = 0;
  const ROLE_ADDED = 1;

  /**
   * {@inheritdoc}
   */
  public static function preCreate(EntityStorageInterface $storage_controller, array &$values) {
    parent::preCreate($storage_controller, $values);
    $values += [
      'uid' => \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return $this->get('name')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setName($name) {
    $this->set('name', $name);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('uid')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->get('uid')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('uid', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('uid', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isPublished() {
    return (bool) $this->getEntityKey('status');
  }

  /**
   * {@inheritdoc}
   */
  public function setPublished($published) {
    $this->set('status', $published ? TRUE : FALSE);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Authored by'))
      ->setDescription(t('The user ID of author of the Role Watchdog entity.'))
      ->setRevisionable(TRUE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setTranslatable(TRUE)
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'author',
        'weight' => 0,
      ])
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 5,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setLabel(t('Name'))
      ->setDescription(t('The name of the Role Watchdog entity.'))
      ->setSettings([
        'max_length' => 50,
        'text_processing' => 0,
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRequired(TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setLabel(t('Publishing status'))
      ->setDescription(t('A boolean indicating whether the Role Watchdog is published.'))
      ->setDefaultValue(TRUE)
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'weight' => -3,
      ]);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created'))
      ->setDescription(t('The time that the entity was created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the entity was last edited.'));

    $fields['action'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Action'))
      ->setDescription(t('The action performed.'))
      ->setReadOnly(TRUE);

    // Role ID changed.
    $fields['field_role_id'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('Role ID'))
      ->setDescription(t('The added or removed user role.'))
      ->setCardinality(BaseFieldDefinition::CARDINALITY_UNLIMITED)
      ->setSetting('target_type', 'user_role')
      ->setSetting('handler', 'default')
      ->setDisplayOptions('form', [
        'type'     => 'entity_reference_autocomplete',
        'weight'   => 1,
        'settings' => [
          'match_operator'    => 'CONTAINS',
          'size'              => '60',
          'autocomplete_type' => 'tags',
          'placeholder'       => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    // User(performed on)
    $fields['field_user_performed_on'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('User(performed on)'))
      ->setDescription(t('The user the action was performed on.'))
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setDisplayOptions('form', [
        'type'     => 'entity_reference_autocomplete',
        'weight'   => 4,
        'settings' => [
          'match_operator'    => 'CONTAINS',
          'size'              => '60',
          'autocomplete_type' => 'tags',
          'placeholder'       => '',
        ],
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE);

    return $fields;
  }

}
