/**
 * @file
 * Google/Rosetta language translations
 */

((Drupal, once, drupalSettings) => {
  Drupal.behaviors.rosetta_translations = {
    attach(context) {
      const cfg = drupalSettings.rosettaTranslation || {};
      const containerId = cfg.containerId || 'rosetta_translate_element';
      once('rosetta_translations', `#${containerId}`, context).forEach(
        function (translationSelect, index) {
          new TranslationSelectWidget(translationSelect);
        },
      );
    },
  };
})(Drupal, once, drupalSettings);

class TranslationSelectWidget {
  constructor(domNode) {
    const cfg = drupalSettings.rosettaTranslation || {};
    const containerId = cfg.containerId || 'rosetta_translate_element';

    // allowed languages (array preferred; fallback to CSV string → array)
    let langList = [];
    if (Array.isArray(cfg.preferredSupportedLanguages)) {
      langList = cfg.preferredSupportedLanguages;
    } else if (typeof cfg.preferredSupportedLanguages === 'string' && cfg.preferredSupportedLanguages.trim().length > 0) {
      langList = cfg.preferredSupportedLanguages
        .split(',')
        .map((s) => s.trim())
        .filter((s) => s.length > 0);
    }

    // search for lang param to set default language used in translation
    let searchParams = new URLSearchParams(location.search);
    if (searchParams.has("lang")) {
      let lang = searchParams.get('lang');
      if (langList.includes(lang)) {
        sessionStorage.setItem("@au5ton/translate-widget/lastLanguage", "\"" + lang + "\"");
      }
    }

    // ignore classes (array preferred; fallback to CSV string → array)
    let ignoreClasses = [];
    if (Array.isArray(cfg.ignoreClasses)) {
      ignoreClasses = cfg.ignoreClasses;
    } else if (typeof cfg.ignoreClasses === 'string' && cfg.ignoreClasses.trim().length > 0) {
      ignoreClasses = cfg.ignoreClasses
        .split(',')
        .map((s) => s.trim())
        .filter((s) => s.length > 0);
    }

    // ignore selectors (array preferred; fallback to CSV string → array)
    let ignoreSelectors = [];
    if (Array.isArray(cfg.ignoreSelectors)) {
      ignoreSelectors = cfg.ignoreSelectors;
    } else if (typeof cfg.ignoreSelectors === 'string' && cfg.ignoreSelectors.trim().length > 0) {
      ignoreSelectors = cfg.ignoreSelectors
        .split(',')
        .map((s) => s.trim())
        .filter((s) => s.length > 0);
    }

    // included attributes (array preferred; fallback to CSV string → array)
    let includedAttributes = [];
    if (Array.isArray(cfg.includedAttributes)) {
      includedAttributes = cfg.includedAttributes;
    } else if (typeof cfg.includedAttributes === 'string' && cfg.includedAttributes.trim().length > 0) {
      includedAttributes = cfg.includedAttributes
        .split(',')
        .map((s) => s.trim())
        .filter((s) => s.length > 0);
    }

    // load rosetta/google translation widget
    const options = {};
    if (cfg.siteName) options.siteName = cfg.siteName;
    options.pageLanguage = cfg.pageLanguage || 'en';
    if (cfg.chunkSize) options.chunkSize = cfg.chunkSize;
    if (langList.length) options.preferredSupportedLanguages = langList;
    if (ignoreClasses.length) options.ignoreClasses = ignoreClasses;
    if (ignoreSelectors.length) options.ignoreSelectors = ignoreSelectors;
    if (cfg.verboseOutput !== undefined) options.verboseOutput = !!cfg.verboseOutput;
    if (typeof cfg.intersectionThreshold === 'number') options.intersectionThreshold = cfg.intersectionThreshold;
    if (cfg.ignoreIntersection !== undefined) options.ignoreIntersection = !!cfg.ignoreIntersection;
    if (cfg.updateDocumentLanguageAttribute !== undefined) options.updateDocumentLanguageAttribute = !!cfg.updateDocumentLanguageAttribute;
    if (includedAttributes.length) options.includedAttributes = includedAttributes;
    if (cfg.buttons) options.buttons = cfg.buttons;
    if (cfg.attributionImageUrl) options.attributionImageUrl = cfg.attributionImageUrl;
    if (cfg.logoImageUrl) options.logoImageUrl = cfg.logoImageUrl;
    if (cfg.endpoints.translate) {
      options.endpoints = {
        translate: cfg.endpoints.translate,
        supportedLanguages: '/modules/custom/brown_health_translations/dist/languages/supportedLanguages.json'
      };
    }

    // Rosetta dist is injected by the module as an external script via html_head.
    const init = () => {
      try {
        if (window.au5ton && typeof window.au5ton.translateWidget === 'function') {
          au5ton.translateWidget(options, containerId);
        } else if (cfg.verboseOutput) {
          console.error('Rosetta dist js not available.');
        }
      } catch (e) {
        if (cfg.verboseOutput) {
          console.error(e);
        }
      }
    };

    if (document.readyState === 'complete') {
      // If document finished loading, delay briefly to allow async script to load.
      setTimeout(init, 0);
    } else {
      // Otherwise, run on window load.
      window.addEventListener('load', () => setTimeout(init, 0));
    }

    // Wire select after widget renders (it may be added asynchronously).
    const attachSelectListener = () => {
      this.selectId = domNode.querySelector('select');
      if (!this.selectId) {
        return false;
      }
      if (cfg.sessionUrlUpdates) {
        this.languageChanged(this.selectId, langList);
        this.selectId.addEventListener('change', () => {
          this.languageChanged(this.selectId, langList);
        });
      }
      return true;
    };

    // Try immediately in case it's already present.
    if (!attachSelectListener()) {
      const observer = new MutationObserver(() => {
        if (attachSelectListener()) {
          observer.disconnect();
        }
      });
      observer.observe(domNode, { childList: true, subtree: true });
      // Safety stop after 5s.
      setTimeout(() => observer.disconnect(), 5000);
    }
  }

  // language changed function to handle session/url param as well as sizing of drop down.
  languageChanged(selectId, langList) {
    // needs a delay to let rosetta do its thing first.
    setTimeout(() => {
      // Get saved data from sessionStorage
      let currentLanguage = sessionStorage.getItem("@au5ton/translate-widget/lastLanguage").replace(/['"]+/g, '');
      let searchParams = new URLSearchParams(location.search);

      // update URL param with new lang
      if (langList.includes(currentLanguage)) {
        if (!searchParams.has("lang")) {
            this.urlLanguageUpdate(currentLanguage)
        } else {
          let lang = searchParams.get('lang');
          if (lang != currentLanguage) {
            this.urlLanguageUpdate(currentLanguage)
          }
        }
      }
    }, "1000");
  }

  // update URL param with new lang
  urlLanguageUpdate(lang) {
    let searchParams = new URLSearchParams(location.search);
    searchParams.set("lang", lang.replace(/['"]+/g, ''));
    window.history.replaceState({}, '', `${location.pathname}?${searchParams}`);
  }
}
