<?php

declare(strict_types=1);

namespace Drupal\rosetta_translation\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\rosetta_translation\Service\RosettaVersionManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Rosetta Translation settings.
 */
final class RosettaTranslationSettingsForm extends ConfigFormBase {

  private RosettaVersionManager $versionManager;

  public function __construct(ConfigFactoryInterface $configFactory, RosettaVersionManager $versionManager) {
    // Parent does not define a constructor in all supported versions; set property directly.
    $this->configFactory = $configFactory;
    $this->versionManager = $versionManager;
  }

  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('config.factory'),
      $container->get('rosetta_translation.version_manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['rosetta_translation.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'rosetta_translation_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('rosetta_translation.settings');
    $tags = $this->versionManager->getTags();

    $form['translate_endpoint'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Translate backend endpoint'),
      '#default_value' => (string) $config->get('translate_endpoint'),
      '#description' => $this->t('URL of the translate backend endpoint. Use Config Split to vary endpoints by environment.'),
      '#required' => TRUE,
    ];

    $form['select_widget_container_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('ID of translation select widget container'),
      '#default_value' => (string) $config->get('select_widget_container_id') ?: 'rosetta_translate_element',
      '#required' => TRUE,
      '#description' => $this->t('ID of the container element that holds the Rosetta translation select widget in your markup.'),
    ];

    $form['page_language'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Page language'),
      '#default_value' => $config->get('page_language') ?? 'en',
      '#description' => $this->t('The source language of the document (e.g., "en"). <a href="https://en.wikipedia.org/wiki/List_of_ISO_639-1_codes" target="_blank">Language code reference</a>'),
      '#required' => TRUE,
    ];

    $form['site_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Site name'),
      '#default_value' => (string) $config->get('site_name'),
      '#description' => $this->t('An optional identifier for the site being translated.'),
    ];

    $form['preferred_supported_languages'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Preferred supported languages'),
      '#default_value' => (string) $config->get('preferred_supported_languages'),
      '#description' => $this->t('Limit the number of supported languages to only those in this list. Comma-separated list of language codes (spaces allowed), e.g., "es, ru, pt-PT". <a href="https://en.wikipedia.org/wiki/List_of_ISO_639-1_codes" target="_blank">Language code reference</a>.'),
      '#maxlength' => 4096,
    ];

    $form['attribution_image_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Attribution image URL'),
      '#default_value' => (string) $config->get('attribution_image_url'),
      '#description' => $this->t('The URL of the image listed below the dropdown element. Used to meet Google Translate API attribution requirements.'),
    ];

    $form['chunk_size'] = [
      '#type' => 'number',
      '#title' => $this->t('Chunk size'),
      '#default_value' => (int) ($config->get('chunk_size') ?? 10),
      '#min' => 1,
      '#description' => $this->t('Translations are done in batches to prevent API errors from translating too many items at once. This is the size of each batch (number of nodes processed).'),
    ];

    $form['intersection_threshold'] = [
      '#type' => 'number',
      '#title' => $this->t('Intersection threshold'),
      '#default_value' => (float) ($config->get('intersection_threshold') ?? 0.0),
      '#step' => 0.01,
      '#min' => 0,
      '#max' => 1,
      '#description' => $this->t('A number between 0.0 and 1.0 specifying the ratio of intersection area to total bounding box area for the observed target. 0.0 means a single visible pixel counts as visible; 1.0 means the entire element must be visible.'),
    ];

    $form['ignore_classes'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Ignore classes'),
      '#default_value' => (string) $config->get('ignore_classes'),
      '#description' => $this->t('Classes to ignore during translation. Comma-separated list of class names (no leading dots). Example: "pac-container, pac-logo".'),
      '#maxlength' => 4096,
    ];

    $form['ignore_selectors'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Ignore selectors'),
      '#default_value' => (string) $config->get('ignore_selectors'),
      '#description' => $this->t('Selectors to ignore during translation. Comma-separated list of CSS selectors (uses Element.matches on candidates).'),
      '#maxlength' => 4096,
    ];

    $form['update_document_language_attribute'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Update document lang attribute'),
      '#default_value' => (bool) $config->get('update_document_language_attribute'),
      '#description' => $this->t('Update the lang attribute on the <html> element as languages change. May affect other tools relying on a specific lang value.'),
    ];

    $form['verbose_output'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Verbose output'),
      '#default_value' => (bool) $config->get('verbose_output'),
      '#description' => $this->t('Print helpful debugging messages to the console.'),
    ];

    $form['show_banner'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show translation banner'),
      '#default_value' => (bool) ($config->get('show_banner') ?? TRUE),
      '#description' => $this->t('Display the translation banner at the top of the page.'),
    ];

    $form['included_attributes'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Included attributes'),
      '#default_value' => (string) $config->get('included_attributes') ?: 'title, placeholder, alt',
      '#description' => $this->t('Comma-separated list of attributes to include in translation. Default: "title, placeholder, alt".'),
      '#maxlength' => 4096,
    ];

    $form['session_url_updates'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable session and URL updates'),
      '#default_value' => (bool) $config->get('session_url_updates'),
      '#description' => $this->t('Update the URL to include the language code and save the language code to sessionStorage so language persists and can be linked to directly to provide instant translation upon loading the page.'),
    ];

    $options = [];
    foreach ($tags as $tag) {
      $options[$tag] = $tag;
    }
    if (empty($options)) {
      $options = ['latest' => 'latest'];
    }
    $form['rosetta_version'] = [
      '#type' => 'select',
      '#title' => $this->t('Rosetta version'),
      '#options' => ['latest' => $this->t('latest (auto)')] + $options,
      '#default_value' => (string) $config->get('rosetta_version') ?: 'latest',
      '#description' => $this->t('Choose a specific tag from GitHub releases, or "latest" to always use the newest release.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $this->configFactory->getEditable('rosetta_translation.settings')
      ->set('page_language', (string) $form_state->getValue('page_language'))
      ->set('site_name', (string) $form_state->getValue('site_name'))
      ->set('preferred_supported_languages', (string) $form_state->getValue('preferred_supported_languages'))
      ->set('attribution_image_url', (string) $form_state->getValue('attribution_image_url'))
      ->set('chunk_size', (int) $form_state->getValue('chunk_size'))
      ->set('intersection_threshold', (float) $form_state->getValue('intersection_threshold'))
      ->set('ignore_classes', (string) $form_state->getValue('ignore_classes'))
      ->set('ignore_selectors', (string) $form_state->getValue('ignore_selectors'))
      ->set('update_document_language_attribute', (bool) $form_state->getValue('update_document_language_attribute'))
      ->set('verbose_output', (bool) $form_state->getValue('verbose_output'))
      ->set('show_banner', (bool) $form_state->getValue('show_banner'))
      ->set('included_attributes', (string) $form_state->getValue('included_attributes'))
      ->set('translate_endpoint', (string) $form_state->getValue('translate_endpoint'))
      ->set('session_url_updates', (bool) $form_state->getValue('session_url_updates'))
      ->set('select_widget_container_id', (string) $form_state->getValue('select_widget_container_id'))
      ->set('rosetta_version', (string) $form_state->getValue('rosetta_version'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
