# Runner API

A generic, lightweight framework for managing and executing background processing tasks in Drupal 10/11+.

## Overview

The Runner API module provides a flexible, extensible framework for handling background task execution in Drupal. It
abstracts away the complexity of task management and processing, allowing developers to focus on implementing
their specific task logic rather than reinventing the wheel.

**Core Features:**

- **Generic Task Framework**: Define custom tasks by implementing a simple interface
- **Flexible Task Management**: Manage task sequencing and execution flow
- **Observer Pattern**: Hook into task lifecycle events for monitoring and logging
- **Async/Sync Execution**: Switch between synchronous and asynchronous processing modes
- **Privilege Isolation**: Run tasks with specific user contexts for permission management
- **Detached Processing**: Execute long-running operations without blocking the main application flow

## Requirements

- **PHP**: 8.3 or higher
- **Drupal**: 10.4 or 11+
- **sm** module: 0.2.0-beta1 (Symfony Messenger integration)

## Installation

1. Add the module to your project:
   ```bash
   composer require drupal/runner
   ```

2. Enable the module:
   ```bash
   drush en runner
   ```

## Architecture

### Core Components

#### Runner Class
The main orchestrator that manages the execution flow of tasks. It:
- Retrieves the next task from a TaskManager
- Dispatches tasks to the message bus
- Handles routing between synchronous and asynchronous transports

```php
$runner->run($task_manager, $run_context);
```

#### RunContext
A context object that carries metadata about the task execution. Features:
- **Observers**: Register observers to monitor task lifecycle events
- **Observer Invoker**: Coordinate observer callbacks
- **Detached Mode**: Flag for asynchronous processing
- **User Privileges**: Execute tasks with a specific user's permissions

Key methods:
- `withObserver(Observer $observer)`: Add an observer
- `getObservers()`: Retrieve all registered observers
- `detached(bool $detached)`: Enable/disable async processing
- `withPrivilegedUserId(int $userId)`: Set execution user context

#### TaskInterface
The interface that all custom tasks must implement:

```php
interface TaskInterface {
  public function run(): ?TaskInterface;
}
```

The `run()` method returns either:
- `null`: Task processing is complete
- A new `TaskInterface` instance if the task requires some sub-task to be executed

#### TaskManager (Abstract)
Define how tasks are sequenced and retrieved. Extend this class to implement your logic:

```php
abstract class TaskManager {
  abstract public function getNextTask(RunContext $run_context): ?TaskInterface;
}
```

The `getNextTask()` method should return:
- A `TaskInterface` instance: Next task to execute
- `null`: No more tasks to execute

#### TaskMessage
A messenger envelope that carries task execution context. Contains:
- `task`: The TaskInterface to execute
- `taskManager`: The TaskManager handling this task's sequence
- `runContext`: Metadata about the execution context

#### TaskOutput
Represents the output from a task execution:

```php
readonly class TaskOutput {
  public string $content;  // Output content
  public string $type;     // Output type (e.g., 'log', 'result')
}
```

#### Observer Pattern
Implement the `Observer` class to hook into task lifecycle events:

```php
class Observer {
  public function onMessage(TaskOutput $output): void {
    // Called when a task produces output
  }

  public function onEnd(): void {
    // Called when task execution completes
  }
}
```

#### Settings
Configurable module settings via `runner.settings` configuration:
- `sync_transport`: Transport name for synchronous execution (default: 'synchronous')
- `async_transport`: Transport name for asynchronous execution (default: 'asynchronous')

## Usage Examples

### Basic Task Implementation

```php
<?php

namespace Drupal\my_module\Task;

use Drupal\runner\Task\TaskInterface;

class MyTask implements TaskInterface {

  public function __construct(
    private string $data,
  ) {}

  public function run(): ?TaskInterface {
    // Do some work
    \Drupal::logger('my_module')->info('Processing: ' . $this->data);

    // No more tasks
    return null;
  }
}
```

### Task Manager Implementation

```php
<?php

namespace Drupal\my_module\Task;

use Drupal\runner\RunContext;
use Drupal\runner\Task\TaskInterface;
use Drupal\runner\Task\TaskManager;

class MyTaskManager extends TaskManager {

  private int $current = 0;
  private array $tasks = [];

  public function __construct(array $tasks) {
    $this->tasks = $tasks;
  }

  public function getNextTask(RunContext $run_context): ?TaskInterface {
    if (!isset($this->tasks[$this->current])) {
      return null;
    }

    return $this->tasks[$this->current++];
  }
}
```

### Running Tasks Synchronously

```php
<?php

use Drupal\runner\Runner;
use Drupal\runner\RunContext;
use Drupal\my_module\Task\MyTask;
use Drupal\my_module\Task\MyTaskManager;

// Inject Runner via dependency injection
$runner = \Drupal::service('Drupal\runner\Runner');

// Create a context (synchronous by default)
$context = new class extends RunContext {};

// Create your tasks and manager
$tasks = [
  new MyTask('Task 1'),
  new MyTask('Task 2'),
];
$manager = new MyTaskManager($tasks);

// Run the tasks
$runner->run($manager, $context);
```

### Running Tasks Asynchronously

```php
<?php

// Create an async context
$context = (new class extends RunContext {})->detached(true);

// Run tasks asynchronously
$runner->run($manager, $context);
```

### Using Observers

```php
<?php

use Drupal\runner\Observer\Observer;
use Drupal\runner\RunContext;
use Drupal\runner\Task\TaskOutput;

class LoggingObserver extends Observer {

  public function onMessage(RunContext $context, TaskOutput $output): void {
    \Drupal::logger('runner')->info($output->content);
  }

  public function onEnd(RunContext $context): void {
    \Drupal::logger('runner')->info('Task execution completed');
  }
}

// Attach observer to context
$context = (new class extends RunContext {})
  ->withObserver(new LoggingObserver());

$runner->run($manager, $context);
```

### Running Tasks with User Privileges

```php
<?php

// Run tasks as a specific user (e.g., privileged admin user)
$context = (new class extends RunContext {})
  ->withPrivilegedUserId(1); // Run as user 1 (admin)

$runner->run($manager, $context);
```

### Sub-Tasks

Tasks can return a sub-task to be executed next:

```php
<?php

namespace Drupal\my_module\Task;

use Drupal\runner\Task\TaskInterface;

class FirstTask implements TaskInterface {

  public function run(): ?TaskInterface {
    \Drupal::logger('my_module')->info('First task running');

    // Return the next task to execute
    return new SubTask();
  }
}

class SubTask implements TaskInterface {

  public function run(): ?TaskInterface {
    \Drupal::logger('my_module')->info('Second task running');

    // No more tasks
    return null;
  }
}
```

## Configuration

Configure the Runner module via `runner.settings` configuration:

```yaml
# config/sync/runner.settings.yml
sync_transport: 'synchronous'
async_transport: 'asynchronous'
```

Or programmatically:

```php
<?php

\Drupal::configFactory()
  ->getEditable('runner.settings')
  ->set('sync_transport', 'synchronous')
  ->set('async_transport', 'asynchronous')
  ->save();
```

## Messenger Integration

The module uses Symfony Messenger for task dispatch. Tasks are wrapped in a `TaskMessage` envelope with:
- **Transport Selection**: Routes to sync or async transport based on context
- **Error Handling**: Captures and logs messenger exceptions

Make sure your Drupal installation has Messenger configured. Default transports:
- `synchronous`: Executes immediately
- `asynchronous`: Queues for later processing (requires a queue transport like DrupalSql, AMQP, Redis, etc.)

## Logging

The module provides a dedicated logger channel: `logger.channel.runner`

Access it via:

```php
<?php

$logger = \Drupal::logger('runner');
$logger->info('Task started');
```

## Development

### Dependencies

Dev dependencies include:
- **PHPUnit**: Unit testing
- **PHPStan**: Static analysis
- **PHPCS**: Code standards checking
- **Drush**: Drupal CLI

### Running Tests

```bash
composer test
```

### Code Standards

Check code style:

```bash
vendor/bin/phpcs src/
```

Fix code style:

```bash
vendor/bin/phpcbf src/
```

### Static Analysis

Run PHPStan:

```bash
vendor/bin/phpstan analyse src/
```
