<?php

declare(strict_types=1);

namespace Drupal\runner\Messenger;

use Drupal\Core\Session\AccountSwitcherInterface;
use Drupal\Core\Session\UserSession;
use Drupal\runner\DispatchTrait;
use Drupal\runner\Settings\Settings;
use Drupal\runner\Settings\SettingsInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\Attribute\Autowire;
use Symfony\Component\Messenger\Attribute\AsMessageHandler;
use Symfony\Component\Messenger\MessageBusInterface;

/**
 * Handler for task messages.
 */
#[AsMessageHandler]
readonly class TaskHandler {

  use DispatchTrait;

  public function __construct(
    private MessageBusInterface $bus,
    #[Autowire(service: Settings::class)]
    private SettingsInterface $settings,
    private AccountSwitcherInterface $accountSwitcher,
    #[Autowire(service: 'logger.channel.runner')]
    private LoggerInterface $logger,
  ) {}

  /**
   * Invokes the task handler.
   */
  public function __invoke(TaskMessage $message): void {
    $task_manager = $message->taskManager;
    $run_context = $message->runContext;

    if ($run_context->isPrivileged()) {
      // Switch to a specific user for privileged operations.
      $this->accountSwitcher->switchTo(
        new UserSession(['uid' => $run_context->getRunAsUserId()]),
      );
    }

    try {
      $sub_task = $message->task->run();

      if ($sub_task !== NULL) {
        $this->dispatchTask($sub_task, $task_manager, $run_context);
      }
      else {
        $next_task = $task_manager->getNextTask($run_context);

        if ($next_task !== NULL) {
          $this->dispatchTask($next_task, $task_manager, $run_context);
        }
      }
    }
    finally {
      if ($run_context->isPrivileged()) {
        // Switch back to the original user.
        $this->accountSwitcher->switchBack();
      }
    }
  }

}
