<?php

declare(strict_types=1);

namespace Drupal\runner;

use Drupal\runner\Observer\Observer;
use Drupal\runner\Observer\ObserverInvoker;
use Drupal\runner\Observer\ObserverInvokerInterface;

/**
 * Run context for tasks.
 */
abstract class RunContext {

  /**
   * @phpstan-var array<Observer>
   */
  protected array $observers = [];

  protected ObserverInvokerInterface $observerInvoker;

  protected bool $detached = FALSE;

  protected int $runAsUserId = 0;

  /**
   * Add an observer.
   */
  public function withObserver(Observer $observer): self {
    $new = clone $this;
    $new->observers[] = $observer;

    return $new;
  }

  /**
   * Add an observer invoker.
   */
  public function withObserverInvoker(ObserverInvokerInterface $invoker): self {
    $new = clone $this;
    $new->observerInvoker = $invoker;

    return $new;
  }

  /**
   * Set detached mode.
   */
  public function detached(bool $detached = TRUE): self {
    $new = clone $this;
    $new->detached = $detached;

    return $new;
  }

  /**
   * Set privileged user ID.
   */
  public function withPrivilegedUserId(int $userId): self {
    $new = clone $this;
    $new->runAsUserId = $userId;

    return $new;
  }

  /**
   * Return the observer invoker.
   */
  public function getObserverInvoker(): ObserverInvokerInterface {
    if (!isset($this->observerInvoker)) {
      $this->observerInvoker = new ObserverInvoker($this->observers);
    }

    return $this->observerInvoker;
  }

  /**
   * Return whether the task is detached.
   */
  public function isDetached(): bool {
    return $this->detached;
  }

  /**
   * Return whether the task is privileged.
   */
  public function isPrivileged(): bool {
    return $this->runAsUserId > 0;
  }

  /**
   * Return the user ID to run the task as.
   */
  public function getRunAsUserId(): int {
    return $this->runAsUserId;
  }

}
