<?php

namespace Drupal\safedelete_menu_report\Service;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\content_moderation\ModerationInformationInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Link;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\Url;
use Drupal\node\Entity\NodeType;

class ReportHelper {

  use StringTranslationTrait;

  protected EntityTypeManagerInterface $etm;
  protected EntityFieldManagerInterface $fieldManager;
  protected ModerationInformationInterface $moderationInfo;

  public function __construct(EntityTypeManagerInterface $etm, EntityFieldManagerInterface $fieldManager, ModerationInformationInterface $moderationInfo) {
    $this->etm = $etm;
    $this->fieldManager = $fieldManager;
    $this->moderationInfo = $moderationInfo;
  }

  public function alterSafedeleteSettingsForm(array &$form, FormStateInterface $form_state): void {
    $config = \Drupal::configFactory()->getEditable('safedelete_menu_report.settings');

    $form['safedelete_menu_report'] = [
      '#type' => 'details',
      '#title' => $this->t('Safedelete Menu Report'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    // Menus list.
    $menu_storage = $this->etm->getStorage('menu');
    $menus = $menu_storage->loadMultiple();
    $menu_options = [];
    foreach ($menus as $menu) {
      $menu_options[$menu->id()] = $menu->label();
    }

    $form['safedelete_menu_report']['menus'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Menus to include in the report'),
      '#options' => $menu_options,
      '#default_value' => array_values($config->get('menus') ?: []),
      '#description' => $this->t('Select one or more menus that the report can target.'),
    ];

    // Bundle select with default logic.
    $bundle_default = $this->resolveDefaultBundle();
    $bundle_options = [];
    foreach (NodeType::loadMultiple() as $type) {
      $bundle_options[$type->id()] = $type->label();
    }

    $form['safedelete_menu_report']['bundle'] = [
      '#type' => 'select',
      '#title' => $this->t('Content type (bundle)'),
      '#options' => $bundle_options,
      '#default_value' => $config->get('bundle') ?: $bundle_default,
      '#required' => TRUE,
      '#ajax' => [
        'callback' => [static::class, 'ajaxModerationRefresh'],
        'wrapper' => 'safedelete-menu-report-ms-wrapper',
      ],
    ];

    // Moderation state select depends on bundle.
    $selected_bundle = $form_state->getValue(['safedelete_menu_report', 'bundle']) ?: ($config->get('bundle') ?: $bundle_default);
    $states = $this->listModerationStatesForBundle($selected_bundle);
    $state_default = $config->get('moderation_state') ?: (isset($states['archived']) ? 'archived' : '');

    $form['safedelete_menu_report']['moderation_state_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'safedelete-menu-report-ms-wrapper'],
    ];

    if (!empty($states)) {
      $form['safedelete_menu_report']['moderation_state_wrapper']['moderation_state'] = [
        '#type' => 'select',
        '#title' => $this->t('Moderation state to include'),
        '#options' => ['' => $this->t('- None -')] + $states,
        '#default_value' => $state_default,
      ];
    }
    else {
      $form['safedelete_menu_report']['moderation_state_wrapper']['info'] = [
        '#type' => 'markup',
        '#markup' => $this->t('The selected content type has no moderation workflow. The report will not filter by moderation state.'),
      ];
    }

    if (isset($form['actions']['submit'])) {
      $form['actions']['submit']['#submit'][] = [static::class, 'settingsSubmit'];
    }
  }

  public static function ajaxModerationRefresh(array &$form, FormStateInterface $form_state) {
    return $form['safedelete_menu_report']['moderation_state_wrapper'];
  }

  public static function settingsSubmit(array $form, FormStateInterface $form_state): void {
    $values = (array) $form_state->getValue('safedelete_menu_report');
    $menus = array_values(array_filter($values['menus'] ?? [], fn($v) => !empty($v)));
    $bundle = $values['bundle'] ?? '';
    $state = $values['moderation_state_wrapper']['moderation_state'] ?? '';

    \Drupal::configFactory()->getEditable('safedelete_menu_report.settings')
      ->set('menus', $menus)
      ->set('bundle', $bundle)
      ->set('moderation_state', $state)
      ->save();
  }

  public function resolveDefaultBundle(): string {
    $types = \Drupal\node\Entity\NodeType::loadMultiple();
    if (isset($types['article'])) { return 'article'; }
    if (isset($types['landing_page'])) { return 'landing_page'; }
    if (!empty($types)) { $first = reset($types); return $first->id(); }
    return 'article';
  }

  public function listModerationStatesForBundle(string $bundle): array {
    $states = [];
    $storage = $this->etm->getStorage('workflow');
    $entities = $storage->loadMultiple();
    foreach ($entities as $workflow) {
      if ($workflow->getTypePlugin()->getPluginId() !== 'content_moderation') { continue; }
      $plugin = $workflow->getTypePlugin();
      $configuration = $plugin->getConfiguration();
      $entity_types = $configuration['entity_types'] ?? [];
      $node_bundles = $entity_types['node'] ?? [];
      if (!in_array($bundle, $node_bundles, TRUE)) { continue; }
      foreach ($plugin->getStates() as $state_id => $state) {
        $states[$state_id] = $state->label();
      }
    }
    asort($states);
    return $states;
  }

  public function getModerationStateLabelForBundle(string $bundle, string $state_id): ?string {
    if ($state_id === '') { return NULL; }
    $storage = $this->etm->getStorage('workflow');
    $workflows = $storage->loadMultiple();
    foreach ($workflows as $workflow) {
      if ($workflow->getTypePlugin()->getPluginId() !== 'content_moderation') { continue; }
      $plugin = $workflow->getTypePlugin();
      $configuration = $plugin->getConfiguration();
      $entity_types = $configuration['entity_types'] ?? [];
      $node_bundles = $entity_types['node'] ?? [];
      if (!in_array($bundle, $node_bundles, TRUE)) { continue; }
      $states = $plugin->getStates();
      if (isset($states[$state_id])) {
        $state = $states[$state_id];
        return method_exists($state, 'label') ? $state->label() : (string) $state_id;
      }
    }
    return NULL;
  }

  protected function extractNodeId($menu_link_content): int {
    $item = $menu_link_content->get('link')->first();
    if ($item) {
      try {
        $url = $item->getUrl();
        if ($url && $url->isRouted() && $url->getRouteName() === 'entity.node.canonical') {
          $params = $url->getRouteParameters();
          if (isset($params['node'])) {
            return (int) $params['node'];
          }
        }
      } catch (\Throwable $e) {}
      $uri = (string) $item->uri;
      if (preg_match('@(^|:)node/(\d+)@', $uri, $m)) {
        return (int) $m[2];
      }
    }
    return 0;
  }

  public function buildReportRows(string $menu_name, string $bundle, string $moderation_state): array {
    $rows = [];
    $langcode = \Drupal::languageManager()->getCurrentLanguage()->getId();

    $mlc_storage = $this->etm->getStorage('menu_link_content');
    $ids = $mlc_storage->getQuery()
      ->condition('menu_name', $menu_name)
      // Include disabled links too.
      // ->condition('enabled', 1)
      ->accessCheck(FALSE)
      ->execute();
    if (empty($ids)) return $rows;

    $links = $mlc_storage->loadMultiple($ids);

    $parent_children = [];
    $link_info = [];
    foreach ($links as $link) {
      $uuid = $link->uuid();
      $mlid = $link->id();
      $parent_pid = (string) ($link->get('parent')->value ?? '');
      $title = $link->label();
      $nid = $this->extractNodeId($link);
      $item = $link->get('link')->first();
      $uri = $item ? (string) $item->uri : '';
      $is_nolink = (strpos($uri, '<nolink>') !== FALSE);

      $link_info[$uuid] = [ 'id' => $mlid,
        'title' => $title,
        'nid' => $nid,
        'is_nolink' => $is_nolink,
        'parent_pid' => $parent_pid
      ];
      if (!isset($parent_children[$parent_pid])) $parent_children[$parent_pid] = [];
      $parent_children[$parent_pid][] = $link;
    }

    // Helper: traverse descendants and collect node ids.
    $descendant_nids = function (string $start_pid) use (&$parent_children, &$link_info): array {
      $stack = [$start_pid];
      $seen = [];
      $nids = [];
      while ($stack) {
        $pid = array_pop($stack);
        if (isset($seen[$pid])) { continue; }
        $seen[$pid] = TRUE;
        if (empty($parent_children[$pid])) { continue; }
        foreach ($parent_children[$pid] as $child_link) {
          $c_uuid = $child_link->uuid();
          $info = $link_info[$c_uuid] ?? NULL;
          if (!$info) { continue; }
          if (!empty($info['nid'])) {
            $nids[$info['nid']] = $info['nid'];
          }
          $stack[] = 'menu_link_content:' . $c_uuid;
        }
      }
      return $nids;
    };

    // Load all referenced nodes once.
    $all_nids = [];
    foreach ($link_info as $li) {
      if (!empty($li['nid'])) { $all_nids[$li['nid']] = $li['nid']; }
    }
    $node_storage = $this->etm->getStorage('node');
    $all_nodes = !empty($all_nids) ? $node_storage->loadMultiple(array_values($all_nids)) : [];

    // Moderation state helper.
    $cms_storage = $this->etm->getStorage('content_moderation_state');
    $state_cache = [];
    $get_state = function($node) use (&$state_cache, $cms_storage) {
      $nid = $node->id();
      if (isset($state_cache[$nid])) return $state_cache[$nid];
      $value = NULL;
      if ($this->moderationInfo->isModeratedEntity($node)) {
        $cms = method_exists($cms_storage, 'loadFromModeratedEntity') ? $cms_storage->loadFromModeratedEntity($node) : NULL;
        if ($cms && $cms->hasField('moderation_state')) {
          $value = $cms->get('moderation_state')->value;
        } elseif ($node->hasField('moderation_state')) {
          $value = $node->get('moderation_state')->value;
        }
      }
      return $state_cache[$nid] = $value;
    };

    foreach ($links as $link) {
      $uuid = $link->uuid();
      $info = $link_info[$uuid];
      $pid = 'menu_link_content:' . $uuid;
      $desc_nids = $descendant_nids($pid);
      // Keep only descendants that are nodes of the configured bundle.
      $desc_nids_bundle = [];
      foreach ($desc_nids as $cnid) {
        if (isset($all_nodes[$cnid]) && $all_nodes[$cnid]->bundle() === $bundle) {
          $desc_nids_bundle[$cnid] = $cnid;
        }
      }

      // Case 1: Parent is a node-link; parent must match bundle & optional state; must have descendants.
      if (!empty($info['nid']) && !empty($desc_nids_bundle)) {
        $pnode = $all_nodes[$info['nid']] ?? NULL;
        if (!$pnode) continue;
        if ($pnode->bundle() !== $bundle) continue;
        if (!empty($moderation_state)) {
          $pstate = $get_state($pnode);
          if ($pstate !== $moderation_state) continue;
        }

        $parent_url = Url::fromUri('internal:/' . $langcode . '/node/' . $info['nid'] . '/edit');
        $rows[] = [ Link::fromTextAndUrl($info['title'], $parent_url)->toString(), (string) $info['nid'] ];

        $child_list = [];
        foreach ($desc_nids as $cnid) {
          $cnode = $all_nodes[$cnid];
          $cbundle = $cnode->bundle();
          $child_url = Url::fromUri('internal:/' . $langcode . '/node/' . $cnid . '/edit');
          $child_list[] = Link::fromTextAndUrl($cnode->label() . ' (' . $cnid . ')' . " - {$cbundle}", $child_url)->toString();
        }
        if (!empty($child_list)) {
          $warning = [
            '#type' => 'container',
            '#attributes' => ['style' => 'padding-left:1.5rem; color: darkorange;'],
            'header' => ['#markup' => '<strong>' . $this->t('Warning, these children need a new parent, please update their parent menu item selection in the node edit form.') . '</strong>'],
            'list' => ['#theme' => 'item_list', '#items' => $child_list],
          ];
          $rows[] = [[ 'data' => $warning, 'colspan' => 2 ]];
        }
      }
      // Case 2: Parent is <nolink>; qualifies if any descendant node link (bundle match) exists.
      elseif (!empty($info['is_nolink']) && !empty($desc_nids)) {
        $child_list = [];
        foreach ($desc_nids as $cnid) {
          if (!isset($all_nodes[$cnid])) { continue; }
          $cnode = $all_nodes[$cnid];
          if ($cnode->bundle() !== $bundle) { continue; }
          $child_url = Url::fromUri('internal:/' . $langcode . '/node/' . $cnid . '/edit');
          $child_list[] = Link::fromTextAndUrl($cnode->label() . ' (' . $cnid . ')', $child_url)->toString();
        }
        if (empty($child_list)) continue;

        $parent_url = Url::fromUri('internal:/admin/structure/menu/item/' . $info['id'] . '/edit');
        $rows[] = [ Link::fromTextAndUrl($info['title'] . ' - <nolink>', $parent_url)->toString(), '—' ];
        $warning = [
          '#type' => 'container',
          '#attributes' => ['style' => 'padding-left:1.5rem; color: darkorange;'],
          'header' => ['#markup' => '<strong>' . $this->t('warning, these children need a new parent, please update their parent menu item selection in the node edit form') . '</strong>'],
          'list' => ['#theme' => 'item_list', '#items' => $child_list],
        ];
        $rows[] = [[ 'data' => $warning, 'colspan' => 2 ]];
      }
    }

    return $rows;
  }

}
