<?php

namespace Drupal\safedelete_menu_report\Service;

use Drupal\Core\Database\Connection;
use Drupal\Core\Language\LanguageManagerInterface;

/**
 * Report service.
 */
final class ReportService {

  protected Connection $database;
  protected LanguageManagerInterface $languageManager;

  public function __construct(Connection $database, LanguageManagerInterface $language_manager) {
    $this->database = $database;
    $this->languageManager = $language_manager;
  }

  /**
   * Find published child nodes under parents whose menu link is disabled.
   *
   * Uses DBTNG with cross-DB CONCATs instead of SUBSTRING_INDEX/CAST.
   * Parents are content parents only (menu_link_content),
   * children are nodes pointed to by the child link__uri (entity:node/* or internal:/node/*).
   *
   * @param string $menu_name
   *   Menu machine name, e.g., 'main'.
   * @param string|null $langcode
   *   Language code; defaults to current if NULL.
   *
   * @return array<int, array<string, scalar|null>>
   *   Rows with keys:
   *     - child_nid
   *     - child_enabled
   *     - parent_nid
   *     - parent_type
   *     - parent_node_published
   *     - parent_enabled
   */
  public function findDisabledParentsWithPublishedChildren(string $menu_name = 'main', ?string $langcode = NULL): array {
    $conn = $this->database;
    $driver = $conn->driver(); // 'mysql' | 'pgsql' | 'sqlite'
    $langcode = $langcode ?: $this->languageManager->getCurrentLanguage()->getId();

    $q = $conn->select('menu_link_content_data', 'c');

    // Build concatenation snippets per driver.
    if ($driver === 'mysql') {
      $childCond =
        '(c.link__uri = CONCAT(:c_i_prefix, n.nid) OR c.link__uri = CONCAT(:c_e_prefix, n.nid))';
      $parentPidCond =
        "c.parent LIKE 'menu_link_content:%' AND c.parent = CONCAT(:p_prefix, pbase.uuid)";
      $parentCond =
        '(pmdat.link__uri = CONCAT(:p_i_prefix, pn.nid) OR pmdat.link__uri = CONCAT(:p_e_prefix, pn.nid))';
      $castArgs = [];
    }
    elseif ($driver === 'pgsql') {
      $childCond =
        '(c.link__uri = :c_i_prefix || n.nid::text OR c.link__uri = :c_e_prefix || n.nid::text)';
      $parentPidCond =
        "c.parent LIKE 'menu_link_content:%' AND c.parent = :p_prefix || pbase.uuid";
      $parentCond =
        '(pmdat.link__uri = :p_i_prefix || pn.nid::text OR pmdat.link__uri = :p_e_prefix || pn.nid::text)';
      $castArgs = [];
    }
    elseif ($driver === 'sqlite') {
      // SQLite concatenates with || and will coerce numbers to text.
      $childCond =
        '(c.link__uri = :c_i_prefix || n.nid OR c.link__uri = :c_e_prefix || n.nid)';
      $parentPidCond =
        "c.parent LIKE 'menu_link_content:%' AND c.parent = :p_prefix || pbase.uuid";
      $parentCond =
        '(pmdat.link__uri = :p_i_prefix || pn.nid OR pmdat.link__uri = :p_e_prefix || pn.nid)';
      $castArgs = [];
    }
    else {
      // Unknown driver: be explicit so failures are obvious.
      throw new \RuntimeException('Unsupported database driver: ' . $driver);
    }

    // Child node join via link__uri.
    $q->leftJoin('node_field_data', 'n', 'n.langcode = c.langcode AND ' . $childCond, [
      ':c_i_prefix' => 'internal:/node/',
      ':c_e_prefix' => 'entity:node/',
    ] + $castArgs);

    // Parent content entity join (only content parents).
    $q->leftJoin('menu_link_content', 'pbase', $parentPidCond, [
      ':p_prefix' => 'menu_link_content:',
    ]);

    // Parent MLC data in same language.
    $q->leftJoin('menu_link_content_data', 'pmdat', 'pmdat.id = pbase.id AND pmdat.langcode = c.langcode');

    // Parent node join via link__uri.
    $q->leftJoin('node_field_data', 'pn', 'pn.langcode = pmdat.langcode AND ' . $parentCond, [
      ':p_i_prefix' => 'internal:/node/',
      ':p_e_prefix' => 'entity:node/',
    ] + $castArgs);

    // SELECT fields.
    $q->addField('n', 'nid', 'child_nid');
    $q->addField('c', 'enabled', 'child_enabled');
    $q->addField('pn', 'nid', 'parent_nid');
    $q->addField('pn', 'type', 'parent_type');
    $q->addField('pn', 'status', 'parent_node_published');
    $q->addField('pmdat', 'enabled', 'parent_enabled');

    // WHERE.
    $q->isNotNull('c.enabled');
    $q->condition('c.menu_name', $menu_name);
    $q->condition('pmdat.enabled', 0);  // parent menu link disabled
    $q->condition('n.status', 1);       // child node published
    $q->condition('c.langcode', $langcode);

    // ORDER BY (matches your original).
    $q->orderBy('pmdat.id', 'ASC');
    $q->orderBy('n.nid', 'ASC');

    $rows = $q->execute()->fetchAll(\PDO::FETCH_ASSOC);

    // Filter out non-node child links (no child_nid).
    return array_values(array_filter($rows, static function (array $r): bool {
      return !empty($r['child_nid']);
    }));
  }

}

