<?php

namespace Drupal\salesforce_push_queue_ui\Controller;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\RedirectCommand;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Messenger\MessengerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for queue operations.
 */
class QueueController extends ControllerBase {

  /**
   * The database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * Constructs a new QueueController object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service.
   */
  public function __construct(Connection $database, MessengerInterface $messenger) {
    $this->database = $database;
    $this->messenger = $messenger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('database'),
      $container->get('messenger')
    );
  }

  /**
   * Resets failure count for a queue item.
   *
   * @param int $item_id
   *   The queue item ID.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Drupal\Core\Ajax\AjaxResponse
   *   The response.
   */
  public function resetFailures($item_id, Request $request) {
    try {
      $updated = $this->database->update('salesforce_push_queue')
        ->fields([
          'failures' => 0,
          'last_failure_message' => NULL,
          'updated' => time(),
        ])
        ->condition('item_id', $item_id)
        ->execute();

      if ($updated) {
        $this->messenger->addStatus($this->t('Reset failure count for queue item @id.', ['@id' => $item_id]));
      }
      else {
        $this->messenger->addWarning($this->t('Queue item @id not found.', ['@id' => $item_id]));
      }
    }
    catch (\Exception $e) {
      $this->messenger->addError($this->t('Error resetting failures for queue item @id: @error', [
        '@id' => $item_id,
        '@error' => $e->getMessage(),
      ]));
    }

    return $this->redirectToQueue($request);
  }

  /**
   * Resets expiration time for a queue item.
   *
   * @param int $item_id
   *   The queue item ID.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Drupal\Core\Ajax\AjaxResponse
   *   The response.
   */
  public function resetExpiration($item_id, Request $request) {
    try {
      $updated = $this->database->update('salesforce_push_queue')
        ->fields([
          'expire' => 0,
          'updated' => time(),
        ])
        ->condition('item_id', $item_id)
        ->execute();

      if ($updated) {
        $this->messenger->addStatus($this->t('Reset expiration time for queue item @id.', ['@id' => $item_id]));
      }
      else {
        $this->messenger->addWarning($this->t('Queue item @id not found.', ['@id' => $item_id]));
      }
    }
    catch (\Exception $e) {
      $this->messenger->addError($this->t('Error resetting expiration for queue item @id: @error', [
        '@id' => $item_id,
        '@error' => $e->getMessage(),
      ]));
    }

    return $this->redirectToQueue($request);
  }

  /**
   * Redirects back to the queue view.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   *
   * @return \Symfony\Component\HttpFoundation\RedirectResponse|\Drupal\Core\Ajax\AjaxResponse
   *   The response.
   */
  protected function redirectToQueue(Request $request) {
    $destination = $request->query->get('destination');

    // If no destination provided, use the current view path
    if (!$destination) {
      $destination = '/admin/content/salesforce-push-queue';
    }

    if ($request->isXmlHttpRequest()) {
      $response = new AjaxResponse();
      $response->addCommand(new RedirectCommand($destination));
      return $response;
    }

    return new RedirectResponse($destination);
  }

}
