<?php

namespace Drupal\salesforce_push_queue_ui\Plugin\views\field;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;

/**
 * A handler to provide operations for queue items.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("salesforce_push_queue_operations")
 */
class QueueOperations extends FieldPluginBase {

  /**
   * {@inheritdoc}
   */
  public function usesGroupBy() {
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    // Do nothing -- to override the parent query.
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['destination'] = ['default' => TRUE];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    $form['destination'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include destination'),
      '#default_value' => $this->options['destination'],
      '#description' => $this->t('Include a "destination" parameter in the link to return the user to the original view upon completing the operation.'),
    ];
    parent::buildOptionsForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    // Get values directly from the row object.
    $item_id = $values->salesforce_push_queue_item_id ?? $values->item_id ?? NULL;
    $failures = $values->salesforce_push_queue_failures ?? $values->failures ?? 0;
    $expire = $values->salesforce_push_queue_expire ?? $values->expire ?? 0;

    $operations = [];

    // Get the current request to preserve query parameters
    $request = \Drupal::request();
    $query_options = [];

    // Always add destination parameter to preserve current view state
    // Build the destination URL with current query parameters
    $destination_path = $request->getPathInfo(); // Gets the path without domain
    $query_params = $request->query->all(); // Gets all query parameters

    // Construct the destination URL
    $destination_url = $destination_path;
    if (!empty($query_params)) {
      $destination_url .= '?' . http_build_query($query_params);
    }

    $query_options['query']['destination'] = $destination_url;

    // Always show reset failures operation for debugging initially.
    if ($failures > 0 || TRUE) {
      $operations['reset_failures'] = [
        'title' => $this->t('Reset failures (@count)', ['@count' => $failures]),
        'url' => Url::fromRoute('salesforce_push_queue_ui.reset_failures', ['item_id' => $item_id], $query_options),
      ];
    }

    // Always show reset expiration operation for debugging initially.
    if ($expire > 0 || TRUE) {
      $operations['reset_expiration'] = [
        'title' => $this->t('Reset expiration (@time)', ['@time' => $expire]),
        'url' => Url::fromRoute('salesforce_push_queue_ui.reset_expiration', ['item_id' => $item_id], $query_options),
      ];
    }

    if (!empty($operations)) {
      return [
        '#type' => 'operations',
        '#links' => $operations,
      ];
    }

    return [
      '#markup' => $this->t('No operations available'),
    ];
  }

  /**
   * Gets a value from the result row.
   *
   * @param \Drupal\views\ResultRow $values
   *   The result row.
   * @param string $field
   *   The field name.
   *
   * @return mixed
   *   The field value.
   */
  public function getValue(ResultRow $values, $field = NULL) {
    if ($field) {
      return $values->{"salesforce_push_queue_$field"} ?? NULL;
    }
    return parent::getValue($values, $field);
  }

}
