<?php

namespace Drupal\salesforce_push_queue_ui\Plugin\views\field;

use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A handler to display Unix timestamps in human-readable format.
 *
 * @ingroup views_field_handlers
 *
 * @ViewsField("salesforce_push_queue_timestamp")
 */
class TimestampField extends FieldPluginBase {

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * Constructs a new TimestampField object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, DateFormatterInterface $date_formatter) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->dateFormatter = $date_formatter;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('date.formatter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function usesGroupBy() {
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['date_format'] = ['default' => 'medium'];
    $options['custom_date_format'] = ['default' => ''];
    $options['timezone'] = ['default' => ''];
    $options['show_relative'] = ['default' => TRUE];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    $form['date_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Date format'),
      '#options' => [
        'small' => $this->t('Small (e.g., 03/09/60)'),
        'medium' => $this->t('Medium (e.g., Wed, 03/09/1960)'),
        'large' => $this->t('Large (e.g., Wednesday, March 9, 1960)'),
        'custom' => $this->t('Custom'),
        'relative' => $this->t('Time ago (e.g., 2 hours ago)'),
      ],
      '#default_value' => $this->options['date_format'],
    ];

    $form['custom_date_format'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom date format'),
      '#description' => $this->t('A user-defined date format. See the <a href="http://php.net/manual/function.date.php">PHP manual</a> for available options.'),
      '#default_value' => $this->options['custom_date_format'],
      '#states' => [
        'visible' => [
          ':input[name="options[date_format]"]' => ['value' => 'custom'],
        ],
      ],
    ];

    $form['timezone'] = [
      '#type' => 'select',
      '#title' => $this->t('Timezone'),
      '#description' => $this->t('Timezone to be used for date output.'),
      '#options' => ['' => $this->t('- Default site/user timezone -')] + system_time_zones(FALSE),
      '#default_value' => $this->options['timezone'],
    ];

    $form['show_relative'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Show relative time'),
      '#description' => $this->t('Show relative time (e.g., "2 hours ago") instead of absolute time when appropriate.'),
      '#default_value' => $this->options['show_relative'],
      '#states' => [
        'invisible' => [
          ':input[name="options[date_format]"]' => ['value' => 'relative'],
        ],
      ],
    ];

    parent::buildOptionsForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    $value = $this->getValue($values);

    if (empty($value) || $value == 0) {
      return $this->t('Never');
    }

    $format = $this->options['date_format'];
    $custom_format = $this->options['custom_date_format'];
    $timezone = !empty($this->options['timezone']) ? $this->options['timezone'] : NULL;

    if ($format === 'relative' || ($this->options['show_relative'] && $format !== 'custom')) {
      // Show relative time.
      $interval = \Drupal::time()->getRequestTime() - $value;

      if ($interval < 60) {
        return $this->t('Just now');
      }
      elseif ($interval < 3600) {
        $minutes = floor($interval / 60);
        return $this->formatPlural($minutes, '1 minute ago', '@count minutes ago');
      }
      elseif ($interval < 86400) {
        $hours = floor($interval / 3600);
        return $this->formatPlural($hours, '1 hour ago', '@count hours ago');
      }
      elseif ($interval < 2592000) {
        $days = floor($interval / 86400);
        return $this->formatPlural($days, '1 day ago', '@count days ago');
      }
      elseif ($interval < 31536000) {
        $months = floor($interval / 2592000);
        return $this->formatPlural($months, '1 month ago', '@count months ago');
      }
      else {
        $years = floor($interval / 31536000);
        return $this->formatPlural($years, '1 year ago', '@count years ago');
      }
    }

    if ($format === 'custom') {
      if ($custom_format) {
        return $this->dateFormatter->format($value, 'custom', $custom_format, $timezone);
      }
      $format = 'medium';
    }

    return $this->dateFormatter->format($value, $format, '', $timezone);
  }

}
