<?php

declare(strict_types=1);

namespace Drupal\sanitize_placeholder_extra\Strategy;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\sanitize_placeholder\Service\ThematicFaker;
use Drupal\sanitize_placeholder\Strategy\StrategyInterface;

/**
 * Strategy that generates a human-friendly internet domain.
 *
 * Produces short two-word domains like "swift-owl.fr" or "ember-labs.net".
 * Avoids hex-like gibberish and respects multibyte safety utilities.
 */
final class DomainStrategy implements StrategyInterface {

  /**
   * Constructs a DomainStrategy.
   *
   * @param \Drupal\sanitize_placeholder\Service\ThematicFaker $faker
   *   The wrapper around Faker used for randomization and locale awareness.
   */
  public function __construct(
    private readonly ThematicFaker $faker,
  ) {}

  /**
   * Get strategy id.
   */
  public function id(): string {
    return 'domain';
  }

  /**
   * Get strategy label.
   */
  public function label(): string {
    return 'Internet domain';
  }

  /**
   * {@inheritdoc}
   */
  public function generate(
    EntityInterface $entity,
    FieldDefinitionInterface $fieldDefinition,
  ): string {
    // Word banks are intentionally small and “brand-ish”.
    $adjectives = [
      'swift', 'brisk', 'clever', 'vivid', 'lunar', 'nova', 'ember',
      'urban', 'zen', 'atlas', 'delta', 'pixel', 'quiet', 'bright',
    ];
    $nouns = [
      'owl', 'fox', 'river', 'harbor', 'spruce', 'forge', 'meadow',
      'studio', 'works', 'labs', 'maple', 'cloud', 'stone', 'sketch',
    ];
    // A small selection of common / EU TLDs (tune to your needs).
    $tlds = ['fr', 'com', 'net', 'org', 'eu', 'pt', 'es', 'de', 'nl', 'pl', 'dk', 'si', 'hu'];

    $left  = $this->slug($this->pick($adjectives));
    $right = $this->slug($this->pick($nouns));
    $tld   = $this->pick($tlds);

    // Build the domain and ensure it is lower-case and ASCII-safe-ish.
    $domain = $left . '-' . $right . '.' . $tld;
    return $this->toAsciiLower($domain);
  }

  /**
   * Pick a random element from an array (uniform).
   *
   * @param array<int, string> $values
   *   The list of candidates.
   *
   * @return string
   *   The selected value.
   *
   * @throws \Exception
   */
  private function pick(array $values): string {
    $max = \count($values) - 1;
    $idx = ($max > 0) ? \random_int(0, $max) : 0;
    return $values[$idx];
  }

  /**
   * Convert to a URL-safe-ish slug fragment (letters, digits, hyphen only).
   *
   * @param string $value
   *   Input string.
   *
   * @return string
   *   Slugged value.
   */
  private function slug(string $value): string {
    // Lowercase (multibyte-safe).
    $value = \mb_strtolower($value);
    // Replace non alnum with hyphen.
    $value = (string) \preg_replace('/[^a-z0-9]+/u', '-', $value);
    // Trim multiple hyphens and edges.
    $value = (string) \preg_replace('/-+/', '-', $value);
    $value = \trim($value, '-');
    return $value === '' ? 'site' : $value;
  }

  /**
   * Lowercase and strip disallowed characters for domains.
   *
   * @param string $value
   *   Input candidate domain.
   *
   * @return string
   *   Lowercased domain with only allowed characters kept.
   */
  private function toAsciiLower(string $value): string {
    $value = \mb_strtolower($value);
    // Allow letters, digits, hyphen and dots (domain separators).
    $value = (string) \preg_replace('/[^a-z0-9\.\-]+/u', '', $value);
    // Collapse any accidental repeats of dots or hyphens.
    $value = (string) \preg_replace('/\-+/', '-', $value);
    $value = (string) \preg_replace('/\.+/', '.', $value);
    $value = \trim($value, '.-');
    return $value === '' ? 'example.com' : $value;
  }

}
