<?php

namespace Drupal\sanitize_placeholder_extra\Strategy;

use Drupal\sanitize_placeholder\Strategy\StrategyInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;

/**
 * Generates a Portuguese NIF (9 digits) with a valid check digit.
 *
 * The first digit is chosen from the typical NIF prefixes and the last digit
 * is computed using the Mod 11 algorithm used by AT.
 */
final class NifPtStrategy implements StrategyInterface {

  /**
   * Get strategy id.
   */
  public function id(): string {
    return 'nif_pt';
  }

  /**
   * Get strategy label.
   */
  public function label(): string {
    return 'Portuguese NIF';
  }

  /**
   * {@inheritdoc}
   *
   * Algorithm:
   *  - Choose first digit from allowed prefixes.
   *  - Fill digits 2..8 randomly.
   *  - Compute check digit: 11 - ((9*d1 + 8*d2 + ... + 2*d8) % 11);
   *    if >= 10 → 0.
   *
   * @throws \Exception
   */
  public function generate(EntityInterface $entity, FieldDefinitionInterface $field): string {
    // Common valid starting digits: 1,2,3,5,6,8,9.
    $firstPool = [1, 2, 3, 5, 6, 8, 9];
    $d = [];
    $d[1] = $firstPool[random_int(0, count($firstPool) - 1)];
    for ($i = 2; $i <= 8; $i++) {
      $d[$i] = random_int(0, 9);
    }

    $sum = 0;
    $weights = [9, 8, 7, 6, 5, 4, 3, 2];
    for ($i = 1; $i <= 8; $i++) {
      $sum += $d[$i] * $weights[$i - 1];
    }
    $check = 11 - ($sum % 11);
    if ($check >= 10) {
      $check = 0;
    }

    return implode('', [
      $d[1], $d[2], $d[3], $d[4], $d[5], $d[6], $d[7], $d[8], $check,
    ]);
  }

}
