<?php

namespace Drupal\sanitize_placeholder_extra\Strategy;

use Drupal\sanitize_placeholder\Strategy\StrategyInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;

/**
 * Generates values using user-defined token patterns.
 *
 * Patterns read from the `sanitize_placeholder_extra.settings` configuration
 * key `patterns` (array of strings). Supported tokens:
 *  - {letters:N}  → N uppercase ASCII letters (A–Z).
 *  - {digits:N}   → N digits (0–9).
 *  - {company}    → a simple, human-friendly company slug (fallback list).
 *
 * Example pattern: "{letters:3}-{digits:4}" → "ABC-4931".
 */
final class PatternStrategy implements StrategyInterface {

  /**
   * {@inheritdoc}
   */
  public function id(): string {
    return 'pattern';
  }

  /**
   * {@inheritdoc}
   */
  public function label(): string {
    return 'Pattern';
  }

  /**
   * Configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  private ConfigFactoryInterface $configFactory;

  /**
   * Construct the strategy.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory service.
   */
  public function __construct(ConfigFactoryInterface $configFactory) {
    $this->configFactory = $configFactory;
  }

  /**
   * {@inheritdoc}
   *
   * @throws \Exception
   */
  public function generate(EntityInterface $entity, FieldDefinitionInterface $field_definition): string {
    $patterns = $this->loadPatterns();
    $pattern = $this->pickPattern($patterns);

    return $this->renderPattern($pattern);
  }

  /**
   * Load the configured patterns, falling back to a sane default.
   *
   * @return string[]
   *   A list of pattern strings.
   */
  private function loadPatterns(): array {
    $config = $this->configFactory->get('sanitize_placeholder_extra.settings');
    $patterns = $config ? $config->get('patterns') : [];
    if (!is_array($patterns) || empty($patterns)) {
      // Default pattern stays close to what you saw earlier (e.g. "BZ-8379").
      return ['{letters:2}-{digits:4}'];
    }
    // Keep only non-empty strings.
    return array_values(array_filter($patterns, static function ($p): bool {
      return is_string($p) && $p !== '';
    }));
  }

  /**
   * Pick one pattern from the list (uniform random).
   *
   * @param string[] $patterns
   *   The available patterns.
   *
   * @return string
   *   The chosen pattern.
   *
   * @throws \Exception
   */
  private function pickPattern(array $patterns): string {
    if (count($patterns) === 1) {
      return $patterns[0];
    }
    $idx = random_int(0, count($patterns) - 1);
    return $patterns[$idx];
  }

  /**
   * Render a pattern by expanding tokens.
   *
   * Supported:
   *  - {letters:N}
   *  - {digits:N}
   *  - {company}
   *
   * @param string $pattern
   *   The raw pattern string.
   *
   * @return string
   *   The rendered value.
   *
   * @throws \Exception
   */
  private function renderPattern(string $pattern): string {
    $out = $pattern;

    // {company}
    $out = str_replace('{company}', $this->randomCompanySlug(), $out);

    // {letters:N}
    $out = preg_replace_callback('/\{letters:(\d+)\}/', static function (array $m): string {
      $n = (int) $m[1];
      return self::randomLetters($n);
    }, $out) ?? $out;

    // {digits:N}
    $out = preg_replace_callback('/\{digits:(\d+)\}/', static function (array $m): string {
      $n = (int) $m[1];
      return self::randomDigits($n);
    }, $out) ?? $out;

    return $out;
  }

  /**
   * Generate N uppercase ASCII letters.
   *
   * @param int $n
   *   Number of letters.
   *
   * @return string
   *   Letters string.
   *
   * @throws \Exception
   */
  private static function randomLetters(int $n): string {
    if ($n <= 0) {
      return '';
    }
    $letters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ';
    $len = strlen($letters);
    $buf = '';
    for ($i = 0; $i < $n; $i++) {
      $buf .= $letters[random_int(0, $len - 1)];
    }
    return $buf;
  }

  /**
   * Generate N digits.
   *
   * @param int $n
   *   Number of digits.
   *
   * @return string
   *   Digits string.
   *
   * @throws \Exception
   */
  private static function randomDigits(int $n): string {
    if ($n <= 0) {
      return '';
    }
    $buf = '';
    for ($i = 0; $i < $n; $i++) {
      $buf .= random_int(0, 9);
    }
    return $buf;
  }

  /**
   * Return a simple, human-friendly "company" slug.
   *
   * This avoids external dependencies and stays deterministic enough for tests.
   *
   * @return string
   *   A slug like "bright-owl" or "pixel-meadow".
   *
   * @throws \Exception
   */
  private function randomCompanySlug(): string {
    $adjectives = [
      'bright', 'clever', 'zen', 'lunar', 'swift', 'ember', 'nova', 'vivid', 'quiet', 'atlas',
    ];
    $nouns = [
      'owl', 'labs', 'spruce', 'forge', 'river', 'harbor', 'meadow', 'works', 'cloud', 'fox',
    ];
    $a = $adjectives[random_int(0, count($adjectives) - 1)];
    $b = $nouns[random_int(0, count($nouns) - 1)];
    return $a . '-' . $b;
  }

}
