<?php

namespace Drupal\sanitize_placeholder_extra\Strategy;

use Drupal\sanitize_placeholder\Strategy\StrategyInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;

/**
 * Generates a valid French SIREN (9-digit) number using the Luhn algorithm.
 */
final class SirenFrStrategy implements StrategyInterface {

  /**
   * Get strategy id.
   */
  public function id(): string {
    return 'siren_fr';
  }

  /**
   * Get strategy label.
   */
  public function label(): string {
    return 'French SIREN (Luhn)';
  }

  /**
   * {@inheritdoc}
   *
   * Creates eight random digits and computes the ninth with Luhn.
   *
   * @throws \Exception
   */
  public function generate(EntityInterface $entity, FieldDefinitionInterface $field): string {
    $digits = [];
    for ($i = 0; $i < 8; $i++) {
      $digits[] = random_int(0, 9);
    }

    // Compute Luhn check digit.
    $sum = 0;
    // For SIREN, apply Luhn on 8 digits (positions from right).
    for ($i = 0; $i < 8; $i++) {
      $d = $digits[7 - $i];
      if ($i % 2 === 0) {
        $d *= 2;
        if ($d > 9) {
          $d -= 9;
        }
      }
      $sum += $d;
    }
    $check = (10 - ($sum % 10)) % 10;

    return implode('', $digits) . (string) $check;
  }

}
